-- ============================================
-- Project Accounting System Database Migration
-- ============================================
-- This migration adds accounting features to the project manager
-- Features: Income/Expense tracking, Contacts, Payment Methods, Invoicing

-- ============================================
-- 1. Add Soft Delete to Projects Table
-- ============================================
ALTER TABLE projects
ADD COLUMN deleted_at TIMESTAMP NULL DEFAULT NULL,
ADD COLUMN deleted_by INT NULL DEFAULT NULL,
ADD INDEX idx_deleted_at (deleted_at);

-- ============================================
-- 2. Create Transactions Table
-- ============================================
CREATE TABLE transactions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NOT NULL,
    transaction_date DATE NOT NULL,
    contact_id INT NULL DEFAULT NULL,
    description TEXT NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    type ENUM('income', 'expense') NOT NULL,
    payment_method_id INT NULL DEFAULT NULL,
    invoice_id INT NULL DEFAULT NULL,
    notes TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
    INDEX idx_project_date (project_id, transaction_date),
    INDEX idx_type (type),
    INDEX idx_contact_id (contact_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 3. Create Contacts Table (Customers/Vendors)
-- ============================================
CREATE TABLE contacts (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NOT NULL,
    name VARCHAR(200) NOT NULL,
    type ENUM('customer', 'vendor', 'both') DEFAULT 'customer',
    email VARCHAR(200) NULL,
    phone VARCHAR(50) NULL,
    address TEXT NULL,
    notes TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
    INDEX idx_project_id (project_id),
    INDEX idx_name (name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 4. Create Payment Methods Table
-- ============================================
CREATE TABLE payment_methods (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NOT NULL,
    name VARCHAR(100) NOT NULL,
    type ENUM('income', 'expense', 'both') DEFAULT 'both',
    is_default BOOLEAN DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
    INDEX idx_project_id (project_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 5. Create Invoices Table
-- ============================================
CREATE TABLE invoices (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NOT NULL,
    transaction_id INT NULL DEFAULT NULL,
    invoice_number VARCHAR(50) NOT NULL UNIQUE,

    -- Company Info (from project)
    company_name VARCHAR(200) NOT NULL,
    company_address TEXT NULL,
    company_phone VARCHAR(50) NULL,
    company_email VARCHAR(200) NULL,

    -- Customer Info
    customer_name VARCHAR(200) NOT NULL,
    customer_email VARCHAR(200) NULL,
    customer_phone VARCHAR(50) NULL,
    customer_address TEXT NULL,

    -- Invoice Details
    invoice_date DATE NOT NULL,
    due_date DATE NULL,
    items JSON NOT NULL,
    subtotal DECIMAL(10, 2) NOT NULL,
    tax_rate DECIMAL(5, 2) DEFAULT 0,
    tax_amount DECIMAL(10, 2) DEFAULT 0,
    total_amount DECIMAL(10, 2) NOT NULL,

    payment_method VARCHAR(100) NULL,
    status ENUM('draft', 'sent', 'paid', 'cancelled') DEFAULT 'draft',
    notes TEXT NULL,

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
    FOREIGN KEY (transaction_id) REFERENCES transactions(id) ON DELETE SET NULL,
    INDEX idx_project_id (project_id),
    INDEX idx_invoice_number (invoice_number),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Migration Complete
-- ============================================
-- Tables created:
-- 1. transactions - Income/Expense tracking
-- 2. contacts - Customer/Vendor management
-- 3. payment_methods - Payment method tracking
-- 4. invoices - Invoice generation and tracking
-- Projects table modified with soft delete columns
