<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['project_id'] ?? 0);
$name = trim($data['name'] ?? '');
$type = $data['type'] ?? 'customer';
$email = trim($data['email'] ?? '');
$phone = trim($data['phone'] ?? '');
$address = trim($data['address'] ?? '');
$notes = trim($data['notes'] ?? '');

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID'], 400);
}

// Verify project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized access'], 403);
}

if (empty($name)) {
    jsonResponse(['success' => false, 'message' => 'Name is required'], 400);
}

if (!in_array($type, ['customer', 'vendor', 'both'])) {
    jsonResponse(['success' => false, 'message' => 'Invalid contact type'], 400);
}

$db = getDB();

try {
    $stmt = $db->prepare("
        INSERT INTO contacts (
            project_id, name, type, email, phone, address, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ");

    $stmt->execute([
        $projectId,
        $name,
        $type,
        $email,
        $phone,
        $address,
        $notes
    ]);

    $contactId = $db->lastInsertId();

    // Get the created contact
    $stmt = $db->prepare("
        SELECT * FROM contacts WHERE id = ?
    ");
    $stmt->execute([$contactId]);
    $contact = $stmt->fetch(PDO::FETCH_ASSOC);

    jsonResponse([
        'success' => true,
        'message' => 'Contact created successfully',
        'contact' => $contact
    ]);

} catch (PDOException $e) {
    error_log("Create Contact Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to create contact'], 500);
}
