<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$contactId = intval($data['id'] ?? 0);
$name = trim($data['name'] ?? '');
$type = $data['type'] ?? 'customer';
$email = trim($data['email'] ?? '');
$phone = trim($data['phone'] ?? '');
$address = trim($data['address'] ?? '');
$notes = trim($data['notes'] ?? '');

// Validation
if ($contactId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid contact ID'], 400);
}

$db = getDB();

try {
    // Verify contact exists and user owns the project
    $stmt = $db->prepare("
        SELECT c.project_id
        FROM contacts c
        JOIN projects p ON c.project_id = p.id
        WHERE c.id = ? AND p.user_id = ?
    ");
    $stmt->execute([$contactId, $userId]);
    $contact = $stmt->fetch();

    if (!$contact) {
        jsonResponse(['success' => false, 'message' => 'Contact not found or unauthorized'], 404);
    }

    if (empty($name)) {
        jsonResponse(['success' => false, 'message' => 'Name is required'], 400);
    }

    if (!in_array($type, ['customer', 'vendor', 'both'])) {
        jsonResponse(['success' => false, 'message' => 'Invalid contact type'], 400);
    }

    // Update contact
    $stmt = $db->prepare("
        UPDATE contacts SET
            name = ?,
            type = ?,
            email = ?,
            phone = ?,
            address = ?,
            notes = ?,
            updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
    ");

    $stmt->execute([
        $name,
        $type,
        $email,
        $phone,
        $address,
        $notes,
        $contactId
    ]);

    // Get the updated contact
    $stmt = $db->prepare("SELECT * FROM contacts WHERE id = ?");
    $stmt->execute([$contactId]);
    $updatedContact = $stmt->fetch(PDO::FETCH_ASSOC);

    jsonResponse([
        'success' => true,
        'message' => 'Contact updated successfully',
        'contact' => $updatedContact
    ]);

} catch (PDOException $e) {
    error_log("Update Contact Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update contact'], 500);
}
