<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['project_id'] ?? 0);
$transactionId = !empty($data['transaction_id']) ? intval($data['transaction_id']) : null;
$companyName = trim($data['company_name'] ?? '');
$companyAddress = trim($data['company_address'] ?? '');
$companyPhone = trim($data['company_phone'] ?? '');
$companyEmail = trim($data['company_email'] ?? '');
$customerName = trim($data['customer_name'] ?? '');
$customerEmail = trim($data['customer_email'] ?? '');
$customerPhone = trim($data['customer_phone'] ?? '');
$customerAddress = trim($data['customer_address'] ?? '');
$invoiceDate = $data['invoice_date'] ?? date('Y-m-d');
$dueDate = $data['due_date'] ?? null;
$items = $data['items'] ?? [];
$taxRate = floatval($data['tax_rate'] ?? 0);
$paymentMethod = trim($data['payment_method'] ?? '');
$notes = trim($data['notes'] ?? '');

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID'], 400);
}

// Verify project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized access'], 403);
}

if (empty($companyName)) {
    jsonResponse(['success' => false, 'message' => 'Company name is required'], 400);
}

if (empty($customerName)) {
    jsonResponse(['success' => false, 'message' => 'Customer name is required'], 400);
}

if (empty($items) || !is_array($items)) {
    jsonResponse(['success' => false, 'message' => 'Invoice items are required'], 400);
}

$db = getDB();

try {
    // Calculate totals
    $subtotal = 0;
    foreach ($items as $item) {
        $subtotal += floatval($item['amount'] ?? 0);
    }

    $taxAmount = ($subtotal * $taxRate) / 100;
    $totalAmount = $subtotal + $taxAmount;

    // Generate invoice number: INV-{project_id}-{year}{month}-{sequence}
    $yearMonth = date('Ym');
    $stmt = $db->prepare("
        SELECT COUNT(*) as count
        FROM invoices
        WHERE project_id = ? AND invoice_number LIKE ?
    ");
    $stmt->execute([$projectId, "INV-{$projectId}-{$yearMonth}-%"]);
    $result = $stmt->fetch();
    $sequence = str_pad($result['count'] + 1, 3, '0', STR_PAD_LEFT);
    $invoiceNumber = "INV-{$projectId}-{$yearMonth}-{$sequence}";

    // Create invoice
    $stmt = $db->prepare("
        INSERT INTO invoices (
            project_id, transaction_id, invoice_number,
            company_name, company_address, company_phone, company_email,
            customer_name, customer_email, customer_phone, customer_address,
            invoice_date, due_date, items, subtotal, tax_rate, tax_amount,
            total_amount, payment_method, status, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");

    $stmt->execute([
        $projectId,
        $transactionId,
        $invoiceNumber,
        $companyName,
        $companyAddress,
        $companyPhone,
        $companyEmail,
        $customerName,
        $customerEmail,
        $customerPhone,
        $customerAddress,
        $invoiceDate,
        $dueDate,
        json_encode($items),
        $subtotal,
        $taxRate,
        $taxAmount,
        $totalAmount,
        $paymentMethod,
        'draft',
        $notes
    ]);

    $invoiceId = $db->lastInsertId();

    // Update transaction with invoice_id if provided
    if ($transactionId) {
        $stmt = $db->prepare("UPDATE transactions SET invoice_id = ? WHERE id = ?");
        $stmt->execute([$invoiceId, $transactionId]);
    }

    // Get the created invoice
    $stmt = $db->prepare("SELECT * FROM invoices WHERE id = ?");
    $stmt->execute([$invoiceId]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
    $invoice['items'] = json_decode($invoice['items'], true);

    jsonResponse([
        'success' => true,
        'message' => 'Invoice created successfully',
        'invoice' => $invoice
    ]);

} catch (PDOException $e) {
    error_log("Create Invoice Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to create invoice'], 500);
}
