<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

if (!isLoggedIn()) {
    header('Location: ../../login.php');
    exit;
}

$userId = getCurrentUserId();
$invoiceId = intval($_GET['id'] ?? 0);

if ($invoiceId <= 0) {
    die('Invalid invoice ID');
}

$db = getDB();

try {
    // Get invoice with verification
    $stmt = $db->prepare("
        SELECT i.*, p.name as project_name
        FROM invoices i
        JOIN projects p ON i.project_id = p.id
        WHERE i.id = ? AND p.user_id = ?
    ");
    $stmt->execute([$invoiceId, $userId]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$invoice) {
        die('Invoice not found or unauthorized');
    }

    // Decode JSON items
    $invoice['items'] = json_decode($invoice['items'], true);

    // For now, generate HTML that can be printed to PDF
    // In production, use TCPDF or similar library
    header('Content-Type: text/html; charset=utf-8');

    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Invoice <?php echo e($invoice['invoice_number']); ?></title>
        <style>
            body { font-family: Arial, sans-serif; margin: 40px; }
            .invoice-header { text-align: center; margin-bottom: 30px; }
            .company-info, .customer-info { margin-bottom: 20px; }
            .invoice-details { margin-bottom: 30px; }
            table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
            th, td { padding: 10px; text-align: left; border-bottom: 1px solid #ddd; }
            th { background-color: #f5f5f5; font-weight: bold; }
            .text-right { text-align: right; }
            .totals { float: right; width: 300px; }
            .totals table { margin-top: 20px; }
            @media print {
                button { display: none; }
            }
        </style>
    </head>
    <body>
        <button onclick="window.print()">Print Invoice</button>

        <div class="invoice-header">
            <h1>INVOICE</h1>
            <h3><?php echo e($invoice['invoice_number']); ?></h3>
        </div>

        <div class="company-info">
            <strong><?php echo e($invoice['company_name']); ?></strong><br>
            <?php if ($invoice['company_address']): ?>
                <?php echo nl2br(e($invoice['company_address'])); ?><br>
            <?php endif; ?>
            <?php if ($invoice['company_phone']): ?>
                Phone: <?php echo e($invoice['company_phone']); ?><br>
            <?php endif; ?>
            <?php if ($invoice['company_email']): ?>
                Email: <?php echo e($invoice['company_email']); ?>
            <?php endif; ?>
        </div>

        <div class="invoice-details">
            <strong>Invoice Date:</strong> <?php echo date('F d, Y', strtotime($invoice['invoice_date'])); ?><br>
            <?php if ($invoice['due_date']): ?>
                <strong>Due Date:</strong> <?php echo date('F d, Y', strtotime($invoice['due_date'])); ?>
            <?php endif; ?>
        </div>

        <div class="customer-info">
            <strong>Bill To:</strong><br>
            <?php echo e($invoice['customer_name']); ?><br>
            <?php if ($invoice['customer_address']): ?>
                <?php echo nl2br(e($invoice['customer_address'])); ?><br>
            <?php endif; ?>
            <?php if ($invoice['customer_email']): ?>
                <?php echo e($invoice['customer_email']); ?><br>
            <?php endif; ?>
            <?php if ($invoice['customer_phone']): ?>
                <?php echo e($invoice['customer_phone']); ?>
            <?php endif; ?>
        </div>

        <table>
            <thead>
                <tr>
                    <th>Description</th>
                    <th class="text-right">Amount</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($invoice['items'] as $item): ?>
                    <tr>
                        <td><?php echo e($item['service'] ?? $item['description'] ?? ''); ?></td>
                        <td class="text-right">$<?php echo number_format($item['amount'], 2); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="totals">
            <table>
                <tr>
                    <td><strong>Subtotal:</strong></td>
                    <td class="text-right">$<?php echo number_format($invoice['subtotal'], 2); ?></td>
                </tr>
                <?php if ($invoice['tax_rate'] > 0): ?>
                    <tr>
                        <td><strong>Tax (<?php echo $invoice['tax_rate']; ?>%):</strong></td>
                        <td class="text-right">$<?php echo number_format($invoice['tax_amount'], 2); ?></td>
                    </tr>
                <?php endif; ?>
                <tr>
                    <td><strong>Total:</strong></td>
                    <td class="text-right"><strong>$<?php echo number_format($invoice['total_amount'], 2); ?></strong></td>
                </tr>
            </table>
        </div>

        <div style="clear: both; padding-top: 40px;">
            <?php if ($invoice['payment_method']): ?>
                <p><strong>Payment Method:</strong> <?php echo e($invoice['payment_method']); ?></p>
            <?php endif; ?>
            <?php if ($invoice['notes']): ?>
                <p><strong>Notes:</strong><br><?php echo nl2br(e($invoice['notes'])); ?></p>
            <?php endif; ?>
        </div>

        <div style="margin-top: 50px; text-align: center; color: #666;">
            <p>Thank you for your business!</p>
        </div>
    </body>
    </html>
    <?php

} catch (PDOException $e) {
    error_log("Generate Invoice PDF Error: " . $e->getMessage());
    die('Failed to generate invoice');
}
