<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$invoiceId = intval($data['id'] ?? 0);
$recipientEmail = trim($data['recipient_email'] ?? '');

if ($invoiceId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid invoice ID'], 400);
}

if (empty($recipientEmail) || !filter_var($recipientEmail, FILTER_VALIDATE_EMAIL)) {
    jsonResponse(['success' => false, 'message' => 'Valid recipient email is required'], 400);
}

$db = getDB();

try {
    // Get invoice with verification
    $stmt = $db->prepare("
        SELECT i.*
        FROM invoices i
        JOIN projects p ON i.project_id = p.id
        WHERE i.id = ? AND p.user_id = ?
    ");
    $stmt->execute([$invoiceId, $userId]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$invoice) {
        jsonResponse(['success' => false, 'message' => 'Invoice not found or unauthorized'], 404);
    }

    // TODO: Implement actual email sending with PHPMailer or similar
    // For now, return a placeholder response

    // Example email implementation (commented out - requires PHPMailer setup):
    /*
    require_once '../../../vendor/phpmailer/phpmailer/src/PHPMailer.php';
    require_once '../../../vendor/phpmailer/phpmailer/src/SMTP.php';

    $mail = new PHPMailer\PHPMailer\PHPMailer();
    $mail->isSMTP();
    $mail->Host = 'smtp.example.com';
    $mail->SMTPAuth = true;
    $mail->Username = 'your@email.com';
    $mail->Password = 'your_password';
    $mail->SMTPSecure = 'tls';
    $mail->Port = 587;

    $mail->setFrom($invoice['company_email'], $invoice['company_name']);
    $mail->addAddress($recipientEmail);
    $mail->Subject = 'Invoice ' . $invoice['invoice_number'];
    $mail->Body = 'Please find attached your invoice.';

    // Attach PDF
    $pdfUrl = APP_URL . '/api/accounts/invoices/generate-pdf.php?id=' . $invoiceId;

    if (!$mail->send()) {
        jsonResponse(['success' => false, 'message' => 'Failed to send email: ' . $mail->ErrorInfo], 500);
    }
    */

    // Update invoice status to 'sent'
    $stmt = $db->prepare("UPDATE invoices SET status = 'sent', updated_at = CURRENT_TIMESTAMP WHERE id = ?");
    $stmt->execute([$invoiceId]);

    jsonResponse([
        'success' => true,
        'message' => 'Email functionality is not yet configured. Invoice status updated to "sent". Please configure email settings to enable sending.',
        'invoice_url' => APP_URL . '/api/accounts/invoices/generate-pdf.php?id=' . $invoiceId
    ]);

} catch (PDOException $e) {
    error_log("Send Invoice Email Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to process email request'], 500);
}
