<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$invoiceId = intval($data['id'] ?? 0);
$status = $data['status'] ?? '';
$companyName = trim($data['company_name'] ?? '');
$companyAddress = trim($data['company_address'] ?? '');
$companyPhone = trim($data['company_phone'] ?? '');
$companyEmail = trim($data['company_email'] ?? '');
$customerName = trim($data['customer_name'] ?? '');
$customerEmail = trim($data['customer_email'] ?? '');
$customerPhone = trim($data['customer_phone'] ?? '');
$customerAddress = trim($data['customer_address'] ?? '');
$invoiceDate = $data['invoice_date'] ?? '';
$dueDate = $data['due_date'] ?? null;
$items = $data['items'] ?? [];
$taxRate = floatval($data['tax_rate'] ?? 0);
$paymentMethod = trim($data['payment_method'] ?? '');
$notes = trim($data['notes'] ?? '');

// Validation
if ($invoiceId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid invoice ID'], 400);
}

$db = getDB();

try {
    // Verify invoice exists and user owns the project
    $stmt = $db->prepare("
        SELECT i.project_id
        FROM invoices i
        JOIN projects p ON i.project_id = p.id
        WHERE i.id = ? AND p.user_id = ?
    ");
    $stmt->execute([$invoiceId, $userId]);
    $invoice = $stmt->fetch();

    if (!$invoice) {
        jsonResponse(['success' => false, 'message' => 'Invoice not found or unauthorized'], 404);
    }

    if (!in_array($status, ['draft', 'sent', 'paid', 'cancelled'])) {
        jsonResponse(['success' => false, 'message' => 'Invalid invoice status'], 400);
    }

    // Calculate totals
    $subtotal = 0;
    foreach ($items as $item) {
        $subtotal += floatval($item['amount'] ?? 0);
    }

    $taxAmount = ($subtotal * $taxRate) / 100;
    $totalAmount = $subtotal + $taxAmount;

    // Update invoice
    $stmt = $db->prepare("
        UPDATE invoices SET
            company_name = ?,
            company_address = ?,
            company_phone = ?,
            company_email = ?,
            customer_name = ?,
            customer_email = ?,
            customer_phone = ?,
            customer_address = ?,
            invoice_date = ?,
            due_date = ?,
            items = ?,
            subtotal = ?,
            tax_rate = ?,
            tax_amount = ?,
            total_amount = ?,
            payment_method = ?,
            status = ?,
            notes = ?,
            updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
    ");

    $stmt->execute([
        $companyName,
        $companyAddress,
        $companyPhone,
        $companyEmail,
        $customerName,
        $customerEmail,
        $customerPhone,
        $customerAddress,
        $invoiceDate,
        $dueDate,
        json_encode($items),
        $subtotal,
        $taxRate,
        $taxAmount,
        $totalAmount,
        $paymentMethod,
        $status,
        $notes,
        $invoiceId
    ]);

    // Get the updated invoice
    $stmt = $db->prepare("SELECT * FROM invoices WHERE id = ?");
    $stmt->execute([$invoiceId]);
    $updatedInvoice = $stmt->fetch(PDO::FETCH_ASSOC);
    $updatedInvoice['items'] = json_decode($updatedInvoice['items'], true);

    jsonResponse([
        'success' => true,
        'message' => 'Invoice updated successfully',
        'invoice' => $updatedInvoice
    ]);

} catch (PDOException $e) {
    error_log("Update Invoice Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update invoice'], 500);
}
