<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['project_id'] ?? 0);
$name = trim($data['name'] ?? '');
$type = $data['type'] ?? 'both';
$isDefault = !empty($data['is_default']);

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID'], 400);
}

// Verify project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized access'], 403);
}

if (empty($name)) {
    jsonResponse(['success' => false, 'message' => 'Name is required'], 400);
}

if (!in_array($type, ['income', 'expense', 'both'])) {
    jsonResponse(['success' => false, 'message' => 'Invalid payment method type'], 400);
}

$db = getDB();

try {
    // If setting as default, unset other defaults first
    if ($isDefault) {
        $stmt = $db->prepare("UPDATE payment_methods SET is_default = 0 WHERE project_id = ?");
        $stmt->execute([$projectId]);
    }

    $stmt = $db->prepare("
        INSERT INTO payment_methods (
            project_id, name, type, is_default
        ) VALUES (?, ?, ?, ?)
    ");

    $stmt->execute([
        $projectId,
        $name,
        $type,
        $isDefault ? 1 : 0
    ]);

    $methodId = $db->lastInsertId();

    // Get the created method
    $stmt = $db->prepare("SELECT * FROM payment_methods WHERE id = ?");
    $stmt->execute([$methodId]);
    $method = $stmt->fetch(PDO::FETCH_ASSOC);
    $method['is_default'] = (bool)$method['is_default'];

    jsonResponse([
        'success' => true,
        'message' => 'Payment method created successfully',
        'payment_method' => $method
    ]);

} catch (PDOException $e) {
    error_log("Create Payment Method Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to create payment method'], 500);
}
