<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$methodId = intval($data['id'] ?? 0);
$name = trim($data['name'] ?? '');
$type = $data['type'] ?? 'both';
$isDefault = !empty($data['is_default']);

// Validation
if ($methodId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid payment method ID'], 400);
}

$db = getDB();

try {
    // Verify method exists and user owns the project
    $stmt = $db->prepare("
        SELECT pm.project_id
        FROM payment_methods pm
        JOIN projects p ON pm.project_id = p.id
        WHERE pm.id = ? AND p.user_id = ?
    ");
    $stmt->execute([$methodId, $userId]);
    $method = $stmt->fetch();

    if (!$method) {
        jsonResponse(['success' => false, 'message' => 'Payment method not found or unauthorized'], 404);
    }

    if (empty($name)) {
        jsonResponse(['success' => false, 'message' => 'Name is required'], 400);
    }

    if (!in_array($type, ['income', 'expense', 'both'])) {
        jsonResponse(['success' => false, 'message' => 'Invalid payment method type'], 400);
    }

    // If setting as default, unset other defaults first
    if ($isDefault) {
        $stmt = $db->prepare("UPDATE payment_methods SET is_default = 0 WHERE project_id = ?");
        $stmt->execute([$method['project_id']]);
    }

    // Update payment method
    $stmt = $db->prepare("
        UPDATE payment_methods SET
            name = ?,
            type = ?,
            is_default = ?
        WHERE id = ?
    ");

    $stmt->execute([
        $name,
        $type,
        $isDefault ? 1 : 0,
        $methodId
    ]);

    // Get the updated method
    $stmt = $db->prepare("SELECT * FROM payment_methods WHERE id = ?");
    $stmt->execute([$methodId]);
    $updatedMethod = $stmt->fetch(PDO::FETCH_ASSOC);
    $updatedMethod['is_default'] = (bool)$updatedMethod['is_default'];

    jsonResponse([
        'success' => true,
        'message' => 'Payment method updated successfully',
        'payment_method' => $updatedMethod
    ]);

} catch (PDOException $e) {
    error_log("Update Payment Method Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update payment method'], 500);
}
