<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

$userId = getCurrentUserId();
$projectId = intval($_GET['project_id'] ?? 0);

if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID'], 400);
}

// Verify project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized access'], 403);
}

$db = getDB();

try {
    // Get filter parameters
    $startDate = $_GET['start_date'] ?? null;
    $endDate = $_GET['end_date'] ?? null;
    $type = $_GET['type'] ?? '';
    $contactId = intval($_GET['contact_id'] ?? 0);
    $methodId = intval($_GET['method_id'] ?? 0);

    // Build WHERE clause for filters
    $whereClauses = ['t.project_id = ?'];
    $params = [$projectId];

    if ($startDate) {
        $whereClauses[] = 't.transaction_date >= ?';
        $params[] = $startDate;
    }

    if ($endDate) {
        $whereClauses[] = 't.transaction_date <= ?';
        $params[] = $endDate;
    }

    if ($type && in_array($type, ['income', 'expense'])) {
        $whereClauses[] = 't.type = ?';
        $params[] = $type;
    }

    if ($contactId > 0) {
        $whereClauses[] = 't.contact_id = ?';
        $params[] = $contactId;
    }

    if ($methodId > 0) {
        $whereClauses[] = 't.payment_method_id = ?';
        $params[] = $methodId;
    }

    $whereClause = implode(' AND ', $whereClauses);

    // Get total income
    $stmt = $db->prepare("
        SELECT COALESCE(SUM(amount), 0) as total_income
        FROM transactions t
        WHERE $whereClause AND t.type = 'income'
    ");
    $stmt->execute($params);
    $incomeData = $stmt->fetch();
    $totalIncome = floatval($incomeData['total_income']);

    // Get total expense
    $stmt = $db->prepare("
        SELECT COALESCE(SUM(amount), 0) as total_expense
        FROM transactions t
        WHERE $whereClause AND t.type = 'expense'
    ");
    $stmt->execute($params);
    $expenseData = $stmt->fetch();
    $totalExpense = floatval($expenseData['total_expense']);

    // Calculate profit
    $profit = $totalIncome - $totalExpense;

    // Get pending invoices count (draft or sent status)
    $stmt = $db->prepare("
        SELECT COUNT(*) as pending_count
        FROM invoices
        WHERE project_id = ? AND status IN ('draft', 'sent')
    ");
    $stmt->execute([$projectId]);
    $invoiceData = $stmt->fetch();
    $pendingInvoices = intval($invoiceData['pending_count']);

    jsonResponse([
        'success' => true,
        'stats' => [
            'total_income' => $totalIncome,
            'total_expense' => $totalExpense,
            'profit' => $profit,
            'pending_invoices' => $pendingInvoices
        ]
    ]);

} catch (PDOException $e) {
    error_log("Account Stats Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to fetch statistics'], 500);
}
