<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['project_id'] ?? 0);
$transactionDate = $data['transaction_date'] ?? '';
$type = $data['type'] ?? '';
$description = trim($data['description'] ?? '');
$amount = floatval($data['amount'] ?? 0);
$contactId = !empty($data['contact_id']) ? intval($data['contact_id']) : null;
$methodId = !empty($data['payment_method_id']) ? intval($data['payment_method_id']) : null;
$notes = trim($data['notes'] ?? '');

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID'], 400);
}

// Verify project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized access'], 403);
}

if (empty($transactionDate)) {
    jsonResponse(['success' => false, 'message' => 'Transaction date is required'], 400);
}

if (!in_array($type, ['income', 'expense'])) {
    jsonResponse(['success' => false, 'message' => 'Invalid transaction type'], 400);
}

if (empty($description)) {
    jsonResponse(['success' => false, 'message' => 'Description is required'], 400);
}

if ($amount <= 0) {
    jsonResponse(['success' => false, 'message' => 'Amount must be greater than 0'], 400);
}

$db = getDB();

try {
    $stmt = $db->prepare("
        INSERT INTO transactions (
            project_id, transaction_date, type, description, amount,
            contact_id, payment_method_id, notes
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");

    $stmt->execute([
        $projectId,
        $transactionDate,
        $type,
        $description,
        $amount,
        $contactId,
        $methodId,
        $notes
    ]);

    $transactionId = $db->lastInsertId();

    // Get the created transaction
    $stmt = $db->prepare("
        SELECT
            t.id,
            t.transaction_date,
            t.description,
            t.amount,
            t.type,
            t.notes,
            c.id as contact_id,
            c.name as contact_name,
            pm.id as method_id,
            pm.name as method_name
        FROM transactions t
        LEFT JOIN contacts c ON t.contact_id = c.id
        LEFT JOIN payment_methods pm ON t.payment_method_id = pm.id
        WHERE t.id = ?
    ");
    $stmt->execute([$transactionId]);
    $transaction = $stmt->fetch(PDO::FETCH_ASSOC);

    jsonResponse([
        'success' => true,
        'message' => 'Transaction created successfully',
        'transaction' => $transaction
    ]);

} catch (PDOException $e) {
    error_log("Create Transaction Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to create transaction'], 500);
}
