<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

$userId = getCurrentUserId();
$projectId = intval($_GET['project_id'] ?? 0);

if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID'], 400);
}

// Verify project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized access'], 403);
}

$db = getDB();

try {
    // Get filter parameters
    $startDate = $_GET['start_date'] ?? null;
    $endDate = $_GET['end_date'] ?? null;
    $type = $_GET['type'] ?? '';
    $contactId = intval($_GET['contact_id'] ?? 0);
    $methodId = intval($_GET['method_id'] ?? 0);
    $search = $_GET['search'] ?? '';

    // Build WHERE clause
    $whereClauses = ['t.project_id = ?'];
    $params = [$projectId];

    if ($startDate) {
        $whereClauses[] = 't.transaction_date >= ?';
        $params[] = $startDate;
    }

    if ($endDate) {
        $whereClauses[] = 't.transaction_date <= ?';
        $params[] = $endDate;
    }

    if ($type && in_array($type, ['income', 'expense'])) {
        $whereClauses[] = 't.type = ?';
        $params[] = $type;
    }

    if ($contactId > 0) {
        $whereClauses[] = 't.contact_id = ?';
        $params[] = $contactId;
    }

    if ($methodId > 0) {
        $whereClauses[] = 't.payment_method_id = ?';
        $params[] = $methodId;
    }

    if ($search) {
        $whereClauses[] = '(t.description LIKE ? OR c.name LIKE ?)';
        $searchParam = '%' . $search . '%';
        $params[] = $searchParam;
        $params[] = $searchParam;
    }

    $whereClause = implode(' AND ', $whereClauses);

    // Get transactions with contact and payment method info
    $stmt = $db->prepare("
        SELECT
            t.id,
            t.transaction_date,
            t.description,
            t.amount,
            t.type,
            t.notes,
            t.invoice_id,
            c.id as contact_id,
            c.name as contact_name,
            pm.id as method_id,
            pm.name as method_name,
            t.created_at,
            t.updated_at
        FROM transactions t
        LEFT JOIN contacts c ON t.contact_id = c.id
        LEFT JOIN payment_methods pm ON t.payment_method_id = pm.id
        WHERE $whereClause
        ORDER BY t.transaction_date DESC, t.created_at DESC
    ");

    $stmt->execute($params);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Format amounts as floats
    foreach ($transactions as &$transaction) {
        $transaction['amount'] = floatval($transaction['amount']);
    }

    jsonResponse([
        'success' => true,
        'transactions' => $transactions,
        'total_count' => count($transactions)
    ]);

} catch (PDOException $e) {
    error_log("Transactions List Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to fetch transactions'], 500);
}
