<?php
require_once '../../../includes/config.php';
require_once '../../../includes/db.php';
require_once '../../../includes/auth.php';
require_once '../../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Invalid request method'], 405);
}

$userId = getCurrentUserId();
$data = json_decode(file_get_contents('php://input'), true);

$transactionId = intval($data['id'] ?? 0);
$transactionDate = $data['transaction_date'] ?? '';
$type = $data['type'] ?? '';
$description = trim($data['description'] ?? '');
$amount = floatval($data['amount'] ?? 0);
$contactId = !empty($data['contact_id']) ? intval($data['contact_id']) : null;
$methodId = !empty($data['payment_method_id']) ? intval($data['payment_method_id']) : null;
$notes = trim($data['notes'] ?? '');

// Validation
if ($transactionId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid transaction ID'], 400);
}

$db = getDB();

try {
    // Verify transaction exists and user owns the project
    $stmt = $db->prepare("
        SELECT t.project_id
        FROM transactions t
        JOIN projects p ON t.project_id = p.id
        WHERE t.id = ? AND p.user_id = ?
    ");
    $stmt->execute([$transactionId, $userId]);
    $transaction = $stmt->fetch();

    if (!$transaction) {
        jsonResponse(['success' => false, 'message' => 'Transaction not found or unauthorized'], 404);
    }

    if (empty($transactionDate)) {
        jsonResponse(['success' => false, 'message' => 'Transaction date is required'], 400);
    }

    if (!in_array($type, ['income', 'expense'])) {
        jsonResponse(['success' => false, 'message' => 'Invalid transaction type'], 400);
    }

    if (empty($description)) {
        jsonResponse(['success' => false, 'message' => 'Description is required'], 400);
    }

    if ($amount <= 0) {
        jsonResponse(['success' => false, 'message' => 'Amount must be greater than 0'], 400);
    }

    // Update transaction
    $stmt = $db->prepare("
        UPDATE transactions SET
            transaction_date = ?,
            type = ?,
            description = ?,
            amount = ?,
            contact_id = ?,
            payment_method_id = ?,
            notes = ?,
            updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
    ");

    $stmt->execute([
        $transactionDate,
        $type,
        $description,
        $amount,
        $contactId,
        $methodId,
        $notes,
        $transactionId
    ]);

    // Get the updated transaction
    $stmt = $db->prepare("
        SELECT
            t.id,
            t.transaction_date,
            t.description,
            t.amount,
            t.type,
            t.notes,
            c.id as contact_id,
            c.name as contact_name,
            pm.id as method_id,
            pm.name as method_name
        FROM transactions t
        LEFT JOIN contacts c ON t.contact_id = c.id
        LEFT JOIN payment_methods pm ON t.payment_method_id = pm.id
        WHERE t.id = ?
    ");
    $stmt->execute([$transactionId]);
    $updatedTransaction = $stmt->fetch(PDO::FETCH_ASSOC);

    jsonResponse([
        'success' => true,
        'message' => 'Transaction updated successfully',
        'transaction' => $updatedTransaction
    ]);

} catch (PDOException $e) {
    error_log("Update Transaction Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update transaction'], 500);
}
