<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

$userId = getCurrentUserId();
$db = getDB();

try {
    // Get project counts by status
    $stmt = $db->prepare("
        SELECT
            COUNT(*) as total,
            SUM(CASE WHEN status = 'working' THEN 1 ELSE 0 END) as working,
            SUM(CASE WHEN status = 'stuck' THEN 1 ELSE 0 END) as stuck,
            SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as success
        FROM projects
        WHERE user_id = ? AND deleted_at IS NULL
    ");
    $stmt->execute([$userId]);
    $projectStats = $stmt->fetch();

    // Get task counts
    $stmt = $db->prepare("
        SELECT
            COUNT(*) as total,
            SUM(CASE WHEN t.status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN t.status = 'working' THEN 1 ELSE 0 END) as working,
            SUM(CASE WHEN t.status = 'stuck' THEN 1 ELSE 0 END) as stuck,
            SUM(CASE WHEN t.status = 'pending' THEN 1 ELSE 0 END) as pending
        FROM tasks t
        JOIN projects p ON t.project_id = p.id
        WHERE p.user_id = ? AND p.deleted_at IS NULL
    ");
    $stmt->execute([$userId]);
    $taskStats = $stmt->fetch();

    // Get recent activity
    $stmt = $db->prepare("
        SELECT * FROM activity_logs
        WHERE user_id = ?
        ORDER BY created_at DESC
        LIMIT 10
    ");
    $stmt->execute([$userId]);
    $recentActivity = $stmt->fetchAll();

    // Get overdue tasks count
    $stmt = $db->prepare("
        SELECT COUNT(*) as overdue
        FROM tasks t
        JOIN projects p ON t.project_id = p.id
        WHERE p.user_id = ?
        AND p.deleted_at IS NULL
        AND t.status != 'completed'
        AND t.due_date IS NOT NULL
        AND t.due_date <= CURDATE()
    ");
    $stmt->execute([$userId]);
    $overdueData = $stmt->fetch();

    jsonResponse([
        'success' => true,
        'stats' => [
            'projects' => [
                'total' => intval($projectStats['total']),
                'working' => intval($projectStats['working']),
                'stuck' => intval($projectStats['stuck']),
                'success' => intval($projectStats['success'])
            ],
            'tasks' => [
                'total' => intval($taskStats['total']),
                'completed' => intval($taskStats['completed']),
                'working' => intval($taskStats['working']),
                'stuck' => intval($taskStats['stuck']),
                'pending' => intval($taskStats['pending']),
                'overdue' => intval($overdueData['overdue'])
            ],
            'recent_activity' => $recentActivity
        ]
    ]);
} catch (PDOException $e) {
    error_log("Dashboard Stats Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to fetch statistics'], 500);
}
