<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'DELETE' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);
$projectId = intval($data['id'] ?? $_GET['id'] ?? 0);

if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID']);
}

$userId = getCurrentUserId();

// Check ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 403);
}

$db = getDB();

try {
    // Get project name before soft deletion
    $stmt = $db->prepare("SELECT name FROM projects WHERE id = ? AND deleted_at IS NULL");
    $stmt->execute([$projectId]);
    $project = $stmt->fetch();

    if (!$project) {
        jsonResponse(['success' => false, 'message' => 'Project not found'], 404);
    }

    // Soft delete project (set deleted_at and deleted_by instead of actual deletion)
    $stmt = $db->prepare("UPDATE projects SET deleted_at = NOW(), deleted_by = ? WHERE id = ?");
    $stmt->execute([$userId, $projectId]);

    // Log activity
    logActivity($userId, 'delete_project', "Deleted project: {$project['name']}", $projectId);

    jsonResponse([
        'success' => true,
        'message' => 'Project deleted successfully. Data retained for recovery if needed.'
    ]);
} catch (PDOException $e) {
    error_log("Delete Project Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to delete project'], 500);
}
