<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

$projectId = intval($_GET['id'] ?? 0);

if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID']);
}

$userId = getCurrentUserId();

// Check ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 403);
}

$db = getDB();

try {
    // Get project with task count
    $stmt = $db->prepare("
        SELECT
            p.*,
            COUNT(t.id) as total_tasks,
            SUM(CASE WHEN t.status = 'completed' THEN 1 ELSE 0 END) as completed_tasks
        FROM projects p
        LEFT JOIN tasks t ON p.id = t.project_id
        WHERE p.id = ?
        GROUP BY p.id
    ");
    $stmt->execute([$projectId]);
    $project = $stmt->fetch();

    if (!$project) {
        jsonResponse(['success' => false, 'message' => 'Project not found'], 404);
    }

    // Calculate completion percentage
    $project['completion_percentage'] = $project['total_tasks'] > 0
        ? round(($project['completed_tasks'] / $project['total_tasks']) * 100)
        : 0;

    jsonResponse([
        'success' => true,
        'project' => $project
    ]);
} catch (PDOException $e) {
    error_log("Get Project Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to fetch project'], 500);
}
