<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

$userId = getCurrentUserId();
$db = getDB();

// Get filter parameters
$status = $_GET['status'] ?? null;
$search = $_GET['search'] ?? null;
$sortBy = $_GET['sort'] ?? 'updated_at';
$sortOrder = $_GET['order'] ?? 'DESC';

// Validate sort parameters
$allowedSort = ['name', 'created_at', 'updated_at', 'status'];
$sortBy = in_array($sortBy, $allowedSort) ? $sortBy : 'updated_at';
$sortOrder = strtoupper($sortOrder) === 'ASC' ? 'ASC' : 'DESC';

try {
    // Build query
    $sql = "
        SELECT
            p.*,
            COUNT(t.id) as total_tasks,
            SUM(CASE WHEN t.status = 'completed' THEN 1 ELSE 0 END) as completed_tasks
        FROM projects p
        LEFT JOIN tasks t ON p.id = t.project_id
        WHERE p.user_id = ? AND p.deleted_at IS NULL
    ";

    $params = [$userId];

    // Add filters
    if ($status && in_array($status, ['working', 'stuck', 'success'])) {
        $sql .= " AND p.status = ?";
        $params[] = $status;
    }

    if ($search) {
        $sql .= " AND (p.name LIKE ? OR p.description LIKE ?)";
        $searchTerm = '%' . $search . '%';
        $params[] = $searchTerm;
        $params[] = $searchTerm;
    }

    $sql .= " GROUP BY p.id";
    $sql .= " ORDER BY p.$sortBy $sortOrder";

    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $projects = $stmt->fetchAll();

    // Calculate completion percentage for each project
    foreach ($projects as &$project) {
        $project['completion_percentage'] = $project['total_tasks'] > 0
            ? round(($project['completed_tasks'] / $project['total_tasks']) * 100)
            : 0;
    }

    jsonResponse([
        'success' => true,
        'projects' => $projects
    ]);
} catch (PDOException $e) {
    error_log("List Projects Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to fetch projects'], 500);
}
