<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['id'] ?? 0);
$name = trim($data['name'] ?? '');
$description = trim($data['description'] ?? '');
$status = $data['status'] ?? null;

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID']);
}

$userId = getCurrentUserId();

// Check ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 403);
}

$db = getDB();

try {
    // Build update query dynamically
    $updates = [];
    $params = [];

    if (!empty($name)) {
        $updates[] = "name = ?";
        $params[] = $name;
    }

    if (isset($description)) {
        $updates[] = "description = ?";
        $params[] = $description;
    }

    if ($status && in_array($status, ['working', 'stuck', 'success'])) {
        $updates[] = "status = ?";
        $params[] = $status;

        // Set completed_at if status is success
        if ($status === 'success') {
            $updates[] = "completed_at = NOW()";
        }
    }

    if (empty($updates)) {
        jsonResponse(['success' => false, 'message' => 'No fields to update']);
    }

    $params[] = $projectId;
    $sql = "UPDATE projects SET " . implode(', ', $updates) . " WHERE id = ?";

    $stmt = $db->prepare($sql);
    $stmt->execute($params);

    // Log activity
    logActivity($userId, 'update_project', "Updated project: $name", $projectId);

    jsonResponse([
        'success' => true,
        'message' => 'Project updated successfully'
    ]);
} catch (PDOException $e) {
    error_log("Update Project Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update project'], 500);
}
