<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['project_id'] ?? 0);
$title = trim($data['title'] ?? '');
$description = trim($data['description'] ?? '');
$notes = trim($data['notes'] ?? '');
$status = $data['status'] ?? 'pending';
$priority = $data['priority'] ?? 'medium';
$dueDate = $data['due_date'] ?? null;

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID']);
}

if (empty($title)) {
    jsonResponse(['success' => false, 'message' => 'Task title is required']);
}

$userId = getCurrentUserId();

// Check project ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 403);
}

// Validate enums
if (!in_array($status, ['pending', 'working', 'stuck', 'completed'])) {
    $status = 'pending';
}

if (!in_array($priority, ['low', 'medium', 'high'])) {
    $priority = 'medium';
}

$db = getDB();

try {
    $stmt = $db->prepare("
        INSERT INTO tasks (project_id, title, description, notes, status, priority, due_date)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([$projectId, $title, $description, $notes, $status, $priority, $dueDate]);

    $taskId = $db->lastInsertId();

    // Log activity
    logActivity($userId, 'create_task', "Created task: $title", $projectId, $taskId);

    jsonResponse([
        'success' => true,
        'message' => 'Task created successfully',
        'task_id' => $taskId
    ]);
} catch (PDOException $e) {
    error_log("Create Task Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to create task'], 500);
}
