<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);

$taskId = intval($data['id'] ?? 0);
$title = trim($data['title'] ?? '');
$description = $data['description'] ?? null;
$notes = $data['notes'] ?? null;
$status = $data['status'] ?? null;
$priority = $data['priority'] ?? null;
$dueDate = $data['due_date'] ?? null;

if ($taskId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid task ID']);
}

$userId = getCurrentUserId();
$db = getDB();

try {
    // Get task and check ownership
    $stmt = $db->prepare("
        SELECT t.*, p.user_id
        FROM tasks t
        JOIN projects p ON t.project_id = p.id
        WHERE t.id = ?
    ");
    $stmt->execute([$taskId]);
    $task = $stmt->fetch();

    if (!$task) {
        jsonResponse(['success' => false, 'message' => 'Task not found'], 404);
    }

    if ($task['user_id'] != $userId) {
        jsonResponse(['success' => false, 'message' => 'Unauthorized'], 403);
    }

    // Build update query dynamically
    $updates = [];
    $params = [];

    if (!empty($title)) {
        $updates[] = "title = ?";
        $params[] = $title;
    }

    if (isset($description)) {
        $updates[] = "description = ?";
        $params[] = $description;
    }

    if (isset($notes)) {
        $updates[] = "notes = ?";
        $params[] = $notes;
    }

    if ($status && in_array($status, ['pending', 'working', 'stuck', 'completed'])) {
        $updates[] = "status = ?";
        $params[] = $status;

        // Set completed_at if status is completed
        if ($status === 'completed') {
            $updates[] = "completed_at = NOW()";
        } elseif ($task['status'] === 'completed' && $status !== 'completed') {
            // Reset completed_at if unmarking as completed
            $updates[] = "completed_at = NULL";
        }
    }

    if ($priority && in_array($priority, ['low', 'medium', 'high'])) {
        $updates[] = "priority = ?";
        $params[] = $priority;
    }

    if (isset($dueDate)) {
        $updates[] = "due_date = ?";
        $params[] = $dueDate ?: null;
    }

    if (empty($updates)) {
        jsonResponse(['success' => false, 'message' => 'No fields to update']);
    }

    $params[] = $taskId;
    $sql = "UPDATE tasks SET " . implode(', ', $updates) . " WHERE id = ?";

    $stmt = $db->prepare($sql);
    $stmt->execute($params);

    // Check if all tasks in the project are completed
    $stmt = $db->prepare("
        SELECT COUNT(*) as total,
               SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed
        FROM tasks
        WHERE project_id = ?
    ");
    $stmt->execute([$task['project_id']]);
    $taskStats = $stmt->fetch();

    // Auto-mark project as success if all tasks are completed
    if ($taskStats['total'] > 0 && $taskStats['total'] == $taskStats['completed']) {
        $stmt = $db->prepare("
            UPDATE projects
            SET status = 'success', completed_at = NOW()
            WHERE id = ? AND status != 'success'
        ");
        $stmt->execute([$task['project_id']]);
    }

    // Log activity
    $titleForLog = !empty($title) ? $title : $task['title'];
    logActivity($userId, 'update_task', "Updated task: $titleForLog", $task['project_id'], $taskId);

    jsonResponse([
        'success' => true,
        'message' => 'Task updated successfully'
    ]);
} catch (PDOException $e) {
    error_log("Update Task Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update task'], 500);
}
