<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);

$projectId = intval($data['project_id'] ?? 0);
$title = trim($data['title'] ?? '');
$url = trim($data['url'] ?? '');
$description = trim($data['description'] ?? '');

// Validation
if ($projectId <= 0) {
    jsonResponse(['success' => false, 'message' => 'Invalid project ID']);
}

if (empty($title)) {
    jsonResponse(['success' => false, 'message' => 'Title is required']);
}

if (empty($url)) {
    jsonResponse(['success' => false, 'message' => 'URL is required']);
}

// Validate URL format
if (!filter_var($url, FILTER_VALIDATE_URL)) {
    jsonResponse(['success' => false, 'message' => 'Invalid URL format']);
}

$userId = getCurrentUserId();

// Check ownership
if (!userOwnsProject($projectId, $userId)) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 403);
}

$db = getDB();

try {
    $stmt = $db->prepare("
        INSERT INTO project_urls (project_id, title, url, description)
        VALUES (?, ?, ?, ?)
    ");
    $stmt->execute([$projectId, $title, $url, $description]);

    $urlId = $db->lastInsertId();

    // Log activity
    logActivity($userId, 'add_url', "Added URL: $title", $projectId);

    jsonResponse([
        'success' => true,
        'message' => 'URL added successfully',
        'url_id' => $urlId
    ]);
} catch (PDOException $e) {
    error_log("Create URL Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to add URL'], 500);
}
