<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);

$currentPassword = $data['currentPassword'] ?? '';
$newPassword = $data['newPassword'] ?? '';

// Validation
if (empty($currentPassword)) {
    jsonResponse(['success' => false, 'message' => 'Current password is required']);
}

if (empty($newPassword)) {
    jsonResponse(['success' => false, 'message' => 'New password is required']);
}

if (strlen($newPassword) < 6) {
    jsonResponse(['success' => false, 'message' => 'New password must be at least 6 characters']);
}

$userId = getCurrentUserId();
$db = getDB();

try {
    // Get current user password hash
    $stmt = $db->prepare("SELECT password_hash FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();

    if (!$user) {
        jsonResponse(['success' => false, 'message' => 'User not found'], 404);
    }

    // Verify current password
    if (!password_verify($currentPassword, $user['password_hash'])) {
        jsonResponse(['success' => false, 'message' => 'Current password is incorrect']);
    }

    // Hash new password
    $newPasswordHash = password_hash($newPassword, PASSWORD_BCRYPT);

    // Update password
    $stmt = $db->prepare("UPDATE users SET password_hash = ? WHERE id = ?");
    $stmt->execute([$newPasswordHash, $userId]);

    // Log activity
    logActivity($userId, 'update_password', "Changed account password");

    jsonResponse([
        'success' => true,
        'message' => 'Password changed successfully'
    ]);

} catch (PDOException $e) {
    error_log("Update Password Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update password'], 500);
}
