<?php
require_once '../../includes/config.php';
require_once '../../includes/db.php';
require_once '../../includes/auth.php';
require_once '../../includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'message' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'message' => 'Method not allowed'], 405);
}

$data = json_decode(file_get_contents('php://input'), true);

$name = trim($data['name'] ?? '');
$username = trim($data['username'] ?? '');
$email = trim($data['email'] ?? '');

// Validation
if (empty($username)) {
    jsonResponse(['success' => false, 'message' => 'Username is required']);
}

if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    jsonResponse(['success' => false, 'message' => 'Valid email is required']);
}

$userId = getCurrentUserId();
$db = getDB();

try {
    // Check if username or email is already taken by another user
    $stmt = $db->prepare("
        SELECT id FROM users
        WHERE (username = ? OR email = ?) AND id != ?
    ");
    $stmt->execute([$username, $email, $userId]);

    if ($stmt->fetch()) {
        jsonResponse(['success' => false, 'message' => 'Username or email already in use']);
    }

    // Update user profile
    $stmt = $db->prepare("
        UPDATE users
        SET name = ?, username = ?, email = ?
        WHERE id = ?
    ");
    $stmt->execute([$name, $username, $email, $userId]);

    // Update session
    $_SESSION['username'] = $username;
    $_SESSION['email'] = $email;

    // Log activity
    logActivity($userId, 'update_profile', "Updated profile information");

    jsonResponse([
        'success' => true,
        'message' => 'Profile updated successfully'
    ]);

} catch (PDOException $e) {
    error_log("Update Profile Error: " . $e->getMessage());
    jsonResponse(['success' => false, 'message' => 'Failed to update profile'], 500);
}
