/**
 * Accounts Management JavaScript
 * Handles income/expense tracking, contacts, payment methods, and invoices
 */

// Global state
let currentFilters = {};
let allContacts = [];
let allPaymentMethods = [];
let currentTransactionId = null;
let currentContactId = null;
let currentMethodId = null;
let deleteCallback = null;

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    loadAccountStats();
    loadTransactions();
    loadContactsForFilters();
    loadMethodsForFilters();
    initializeEventListeners();
    setDefaultDates();
});

// Initialize event listeners
function initializeEventListeners() {
    // Transaction buttons
    document.getElementById('addTransactionBtn').addEventListener('click', showAddTransactionModal);
    document.getElementById('saveTransactionBtn').addEventListener('click', saveTransaction);

    // Contact buttons
    document.getElementById('manageContactsBtn').addEventListener('click', showContactsModal);
    document.getElementById('addContactBtn').addEventListener('click', showAddContactModal);
    document.getElementById('saveContactBtn').addEventListener('click', saveContact);

    // Payment method buttons
    document.getElementById('manageMethodsBtn').addEventListener('click', showMethodsModal);
    document.getElementById('addMethodBtn').addEventListener('click', showAddMethodModal);
    document.getElementById('saveMethodBtn').addEventListener('click', saveMethod);

    // Filter buttons
    document.getElementById('applyFilters').addEventListener('click', applyFilters);
    document.getElementById('clearFilters').addEventListener('click', clearFilters);

    // Search filter with debounce
    let searchTimeout;
    document.getElementById('searchFilter').addEventListener('input', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => applyFilters(), 500);
    });

    // Delete confirmation
    document.getElementById('confirmDeleteBtn').addEventListener('click', confirmDelete);

    // Invoice buttons
    document.getElementById('addInvoiceItemBtn').addEventListener('click', () => addInvoiceItem());
    document.getElementById('saveInvoiceBtn').addEventListener('click', saveInvoice);
    document.getElementById('invoiceTaxRate').addEventListener('input', calculateInvoiceTotals);
}

// Set default dates (last 30 days)
function setDefaultDates() {
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - 30);

    document.getElementById('endDate').valueAsDate = endDate;
    document.getElementById('startDate').valueAsDate = startDate;
}

// Load account statistics
async function loadAccountStats() {
    try {
        const params = new URLSearchParams({
            project_id: PROJECT_ID,
            ...currentFilters
        });

        const response = await fetch(`api/accounts/stats.php?${params}`);
        const data = await response.json();

        if (data.success) {
            renderStats(data.stats);
        } else {
            showAlert('Failed to load statistics', 'danger');
        }
    } catch (error) {
        console.error('Error loading stats:', error);
        showAlert('Error loading statistics', 'danger');
    }
}

// Render statistics cards
function renderStats(stats) {
    document.getElementById('totalIncome').textContent = '$' + formatNumber(stats.total_income);
    document.getElementById('totalExpense').textContent = '$' + formatNumber(stats.total_expense);
    document.getElementById('netProfit').textContent = '$' + formatNumber(stats.profit);
    document.getElementById('pendingInvoices').textContent = stats.pending_invoices;

    // Update summary footer
    document.getElementById('summaryIncome').textContent = '$' + formatNumber(stats.total_income);
    document.getElementById('summaryExpense').textContent = '$' + formatNumber(stats.total_expense);
    const netSpan = document.getElementById('summaryNet');
    netSpan.textContent = '$' + formatNumber(stats.profit);
    netSpan.className = stats.profit >= 0 ? 'text-success' : 'text-danger';
}

// Load transactions
async function loadTransactions() {
    try {
        const params = new URLSearchParams({
            project_id: PROJECT_ID,
            ...currentFilters
        });

        const response = await fetch(`api/accounts/transactions/list.php?${params}`);
        const data = await response.json();

        if (data.success) {
            renderTransactions(data.transactions);
        } else {
            showAlert('Failed to load transactions', 'danger');
        }
    } catch (error) {
        console.error('Error loading transactions:', error);
        showAlert('Error loading transactions', 'danger');
    }
}

// Render transactions table
function renderTransactions(transactions) {
    const tbody = document.getElementById('transactionsList');

    if (transactions.length === 0) {
        tbody.innerHTML = '<tr><td colspan="8" class="text-center py-4">No transactions found</td></tr>';
        return;
    }

    tbody.innerHTML = transactions.map(t => {
        const isIncome = t.type === 'income';
        return `
            <tr>
                <td>${formatDate(t.transaction_date)}</td>
                <td>${escapeHtml(t.contact_name || '-')}</td>
                <td>${escapeHtml(t.description)}</td>
                <td class="text-end text-success fw-bold">
                    ${isIncome ? '$' + formatNumber(t.amount) : '-'}
                </td>
                <td>${isIncome ? escapeHtml(t.method_name || '-') : '-'}</td>
                <td class="text-end text-danger fw-bold">
                    ${!isIncome ? '$' + formatNumber(t.amount) : '-'}
                </td>
                <td>${!isIncome ? escapeHtml(t.method_name || '-') : '-'}</td>
                <td>
                    <button class="btn btn-sm btn-outline-primary" onclick="editTransaction(${t.id})" title="Edit">
                        <i class="bi bi-pencil"></i>
                    </button>
                    ${isIncome && !t.invoice_id ? `
                        <button class="btn btn-sm btn-outline-success" onclick="generateInvoice(${t.id})" title="Generate Invoice">
                            <i class="bi bi-file-earmark-text"></i>
                        </button>
                    ` : ''}
                    ${t.invoice_id ? `
                        <a href="api/accounts/invoices/generate-pdf.php?id=${t.invoice_id}" target="_blank" class="btn btn-sm btn-outline-info" title="View Invoice">
                            <i class="bi bi-eye"></i>
                        </a>
                    ` : ''}
                    <button class="btn btn-sm btn-outline-danger" onclick="deleteTransaction(${t.id})" title="Delete">
                        <i class="bi bi-trash"></i>
                    </button>
                </td>
            </tr>
        `;
    }).join('');
}

// Apply filters
function applyFilters() {
    currentFilters = {
        start_date: document.getElementById('startDate').value || undefined,
        end_date: document.getElementById('endDate').value || undefined,
        type: document.getElementById('typeFilter').value || undefined,
        contact_id: document.getElementById('contactFilter').value || undefined,
        method_id: document.getElementById('methodFilter').value || undefined,
        search: document.getElementById('searchFilter').value || undefined
    };

    // Remove undefined values
    Object.keys(currentFilters).forEach(key =>
        currentFilters[key] === undefined && delete currentFilters[key]
    );

    loadAccountStats();
    loadTransactions();
}

// Clear filters
function clearFilters() {
    document.getElementById('startDate').value = '';
    document.getElementById('endDate').value = '';
    document.getElementById('typeFilter').value = '';
    document.getElementById('contactFilter').value = '';
    document.getElementById('methodFilter').value = '';
    document.getElementById('searchFilter').value = '';

    currentFilters = {};
    setDefaultDates();
    applyFilters();
}

// Show add transaction modal
function showAddTransactionModal() {
    currentTransactionId = null;
    document.getElementById('transactionModalTitle').textContent = 'Add Transaction';
    document.getElementById('transactionForm').reset();
    document.getElementById('transactionId').value = '';
    document.getElementById('transactionDate').valueAsDate = new Date();

    loadContactsForTransaction();
    loadMethodsForTransaction();

    const modal = new bootstrap.Modal(document.getElementById('transactionModal'));
    modal.show();
}

// Edit transaction
async function editTransaction(id) {
    try {
        const response = await fetch(`api/accounts/transactions/get.php?id=${id}`);
        const data = await response.json();

        if (data.success) {
            currentTransactionId = id;
            document.getElementById('transactionModalTitle').textContent = 'Edit Transaction';
            document.getElementById('transactionId').value = data.transaction.id;
            document.getElementById('transactionDate').value = data.transaction.transaction_date;
            document.getElementById('transactionType').value = data.transaction.type;
            document.getElementById('transactionDescription').value = data.transaction.description;
            document.getElementById('transactionAmount').value = data.transaction.amount;
            document.getElementById('transactionNotes').value = data.transaction.notes || '';

            await loadContactsForTransaction();
            await loadMethodsForTransaction();

            document.getElementById('transactionContact').value = data.transaction.contact_id || '';
            document.getElementById('transactionMethod').value = data.transaction.method_id || '';

            const modal = new bootstrap.Modal(document.getElementById('transactionModal'));
            modal.show();
        } else {
            showAlert('Failed to load transaction', 'danger');
        }
    } catch (error) {
        console.error('Error loading transaction:', error);
        showAlert('Error loading transaction', 'danger');
    }
}

// Save transaction
async function saveTransaction() {
    const transactionId = document.getElementById('transactionId').value;
    const formData = {
        project_id: PROJECT_ID,
        transaction_date: document.getElementById('transactionDate').value,
        type: document.getElementById('transactionType').value,
        description: document.getElementById('transactionDescription').value,
        amount: parseFloat(document.getElementById('transactionAmount').value),
        contact_id: document.getElementById('transactionContact').value || null,
        payment_method_id: document.getElementById('transactionMethod').value || null,
        notes: document.getElementById('transactionNotes').value
    };

    if (!formData.transaction_date || !formData.description || !formData.amount) {
        showAlert('Please fill in all required fields', 'warning');
        return;
    }

    try {
        const url = transactionId ?
            'api/accounts/transactions/update.php' :
            'api/accounts/transactions/create.php';

        if (transactionId) {
            formData.id = transactionId;
        }

        const response = await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.success) {
            showAlert(data.message, 'success');
            bootstrap.Modal.getInstance(document.getElementById('transactionModal')).hide();
            loadAccountStats();
            loadTransactions();
        } else {
            showAlert(data.message, 'danger');
        }
    } catch (error) {
        console.error('Error saving transaction:', error);
        showAlert('Error saving transaction', 'danger');
    }
}

// Delete transaction
function deleteTransaction(id) {
    deleteCallback = async () => {
        try {
            const response = await fetch('api/accounts/transactions/delete.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ id: id })
            });

            const data = await response.json();

            if (data.success) {
                showAlert(data.message, 'success');
                loadAccountStats();
                loadTransactions();
            } else {
                showAlert(data.message, 'danger');
            }
        } catch (error) {
            console.error('Error deleting transaction:', error);
            showAlert('Error deleting transaction', 'danger');
        }
    };

    document.getElementById('deleteMessage').textContent =
        'Are you sure you want to delete this transaction? This action cannot be undone.';

    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

// Load contacts for filters
async function loadContactsForFilters() {
    try {
        const response = await fetch(`api/accounts/contacts/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            const select = document.getElementById('contactFilter');
            select.innerHTML = '<option value="">All Contacts</option>' +
                data.contacts.map(c => `<option value="${c.id}">${escapeHtml(c.name)}</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading contacts:', error);
    }
}

// Load contacts for transaction form
async function loadContactsForTransaction() {
    try {
        const response = await fetch(`api/accounts/contacts/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            allContacts = data.contacts;
            const select = document.getElementById('transactionContact');
            select.innerHTML = '<option value="">Select Contact...</option>' +
                data.contacts.map(c => `<option value="${c.id}">${escapeHtml(c.name)} (${c.type})</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading contacts:', error);
    }
}

// Load payment methods for filters
async function loadMethodsForFilters() {
    try {
        const response = await fetch(`api/accounts/payment-methods/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            const select = document.getElementById('methodFilter');
            select.innerHTML = '<option value="">All Methods</option>' +
                data.payment_methods.map(m => `<option value="${m.id}">${escapeHtml(m.name)}</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading payment methods:', error);
    }
}

// Load payment methods for transaction form
async function loadMethodsForTransaction() {
    try {
        const response = await fetch(`api/accounts/payment-methods/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            allPaymentMethods = data.payment_methods;
            const select = document.getElementById('transactionMethod');
            select.innerHTML = '<option value="">Select Method...</option>' +
                data.payment_methods.map(m => `<option value="${m.id}">${escapeHtml(m.name)}</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading payment methods:', error);
    }
}

// Show contacts modal
async function showContactsModal() {
    try {
        const response = await fetch(`api/accounts/contacts/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            renderContactsList(data.contacts);
            const modal = new bootstrap.Modal(document.getElementById('contactsModal'));
            modal.show();
        } else {
            showAlert('Failed to load contacts', 'danger');
        }
    } catch (error) {
        console.error('Error loading contacts:', error);
        showAlert('Error loading contacts', 'danger');
    }
}

// Render contacts list
function renderContactsList(contacts) {
    const container = document.getElementById('contactsList');

    if (contacts.length === 0) {
        container.innerHTML = '<p class="text-center text-muted">No contacts yet</p>';
        return;
    }

    container.innerHTML = contacts.map(c => `
        <div class="card mb-2">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <h6 class="mb-1">${escapeHtml(c.name)}</h6>
                        <span class="badge bg-secondary">${c.type}</span>
                        ${c.email ? `<p class="mb-0 small"><i class="bi bi-envelope"></i> ${escapeHtml(c.email)}</p>` : ''}
                        ${c.phone ? `<p class="mb-0 small"><i class="bi bi-telephone"></i> ${escapeHtml(c.phone)}</p>` : ''}
                    </div>
                    <div>
                        <button class="btn btn-sm btn-outline-primary me-1" onclick="editContact(${c.id})">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="deleteContact(${c.id})">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `).join('');
}

// Show add contact modal
function showAddContactModal() {
    currentContactId = null;
    document.getElementById('contactFormModalTitle').textContent = 'Add Contact';
    document.getElementById('contactForm').reset();
    document.getElementById('contactId').value = '';

    const modal = new bootstrap.Modal(document.getElementById('contactFormModal'));
    modal.show();
}

// Edit contact
async function editContact(id) {
    try {
        const response = await fetch(`api/accounts/contacts/get.php?id=${id}`);
        const data = await response.json();

        if (data.success) {
            currentContactId = id;
            document.getElementById('contactFormModalTitle').textContent = 'Edit Contact';
            document.getElementById('contactId').value = data.contact.id;
            document.getElementById('contactName').value = data.contact.name;
            document.getElementById('contactType').value = data.contact.type;
            document.getElementById('contactEmail').value = data.contact.email || '';
            document.getElementById('contactPhone').value = data.contact.phone || '';
            document.getElementById('contactAddress').value = data.contact.address || '';
            document.getElementById('contactNotes').value = data.contact.notes || '';

            const modal = new bootstrap.Modal(document.getElementById('contactFormModal'));
            modal.show();
        } else {
            showAlert('Failed to load contact', 'danger');
        }
    } catch (error) {
        console.error('Error loading contact:', error);
        showAlert('Error loading contact', 'danger');
    }
}

// Save contact
async function saveContact() {
    const contactId = document.getElementById('contactId').value;
    const formData = {
        project_id: PROJECT_ID,
        name: document.getElementById('contactName').value,
        type: document.getElementById('contactType').value,
        email: document.getElementById('contactEmail').value,
        phone: document.getElementById('contactPhone').value,
        address: document.getElementById('contactAddress').value,
        notes: document.getElementById('contactNotes').value
    };

    if (!formData.name) {
        showAlert('Please enter a name', 'warning');
        return;
    }

    try {
        const url = contactId ?
            'api/accounts/contacts/update.php' :
            'api/accounts/contacts/create.php';

        if (contactId) {
            formData.id = contactId;
        }

        const response = await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.success) {
            showAlert(data.message, 'success');
            bootstrap.Modal.getInstance(document.getElementById('contactFormModal')).hide();
            showContactsModal(); // Refresh list
            loadContactsForFilters();
        } else {
            showAlert(data.message, 'danger');
        }
    } catch (error) {
        console.error('Error saving contact:', error);
        showAlert('Error saving contact', 'danger');
    }
}

// Delete contact
function deleteContact(id) {
    deleteCallback = async () => {
        try {
            const response = await fetch('api/accounts/contacts/delete.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ id: id })
            });

            const data = await response.json();

            if (data.success) {
                showAlert(data.message, 'success');
                showContactsModal(); // Refresh list
                loadContactsForFilters();
            } else {
                showAlert(data.message, 'danger');
            }
        } catch (error) {
            console.error('Error deleting contact:', error);
            showAlert('Error deleting contact', 'danger');
        }
    };

    document.getElementById('deleteMessage').textContent =
        'Are you sure you want to delete this contact?';

    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

// Show methods modal
async function showMethodsModal() {
    try {
        const response = await fetch(`api/accounts/payment-methods/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            renderMethodsList(data.payment_methods);
            const modal = new bootstrap.Modal(document.getElementById('methodsModal'));
            modal.show();
        } else {
            showAlert('Failed to load payment methods', 'danger');
        }
    } catch (error) {
        console.error('Error loading payment methods:', error);
        showAlert('Error loading payment methods', 'danger');
    }
}

// Render methods list
function renderMethodsList(methods) {
    const container = document.getElementById('methodsList');

    if (methods.length === 0) {
        container.innerHTML = '<p class="text-center text-muted">No payment methods yet</p>';
        return;
    }

    container.innerHTML = methods.map(m => `
        <div class="card mb-2">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="mb-1">
                            ${escapeHtml(m.name)}
                            ${m.is_default ? '<span class="badge bg-success ms-2">Default</span>' : ''}
                        </h6>
                        <span class="badge bg-secondary">${m.type}</span>
                    </div>
                    <div>
                        <button class="btn btn-sm btn-outline-primary me-1" onclick="editMethod(${m.id})">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="deleteMethod(${m.id})">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `).join('');
}

// Show add method modal
function showAddMethodModal() {
    currentMethodId = null;
    document.getElementById('methodFormModalTitle').textContent = 'Add Payment Method';
    document.getElementById('methodForm').reset();
    document.getElementById('methodId').value = '';

    const modal = new bootstrap.Modal(document.getElementById('methodFormModal'));
    modal.show();
}

// Edit method
async function editMethod(id) {
    try {
        const method = allPaymentMethods.find(m => m.id === id);
        if (!method) {
            // Fetch if not in cache
            const response = await fetch(`api/accounts/payment-methods/list.php?project_id=${PROJECT_ID}`);
            const data = await response.json();
            if (data.success) {
                const found = data.payment_methods.find(m => m.id === id);
                if (found) {
                    fillMethodForm(found);
                }
            }
        } else {
            fillMethodForm(method);
        }
    } catch (error) {
        console.error('Error loading method:', error);
        showAlert('Error loading payment method', 'danger');
    }
}

function fillMethodForm(method) {
    currentMethodId = method.id;
    document.getElementById('methodFormModalTitle').textContent = 'Edit Payment Method';
    document.getElementById('methodId').value = method.id;
    document.getElementById('methodName').value = method.name;
    document.getElementById('methodType').value = method.type;
    document.getElementById('methodIsDefault').checked = method.is_default;

    const modal = new bootstrap.Modal(document.getElementById('methodFormModal'));
    modal.show();
}

// Save method
async function saveMethod() {
    const methodId = document.getElementById('methodId').value;
    const formData = {
        project_id: PROJECT_ID,
        name: document.getElementById('methodName').value,
        type: document.getElementById('methodType').value,
        is_default: document.getElementById('methodIsDefault').checked
    };

    if (!formData.name) {
        showAlert('Please enter a name', 'warning');
        return;
    }

    try {
        const url = methodId ?
            'api/accounts/payment-methods/update.php' :
            'api/accounts/payment-methods/create.php';

        if (methodId) {
            formData.id = methodId;
        }

        const response = await fetch(url, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.success) {
            showAlert(data.message, 'success');
            bootstrap.Modal.getInstance(document.getElementById('methodFormModal')).hide();
            showMethodsModal(); // Refresh list
            loadMethodsForFilters();
        } else {
            showAlert(data.message, 'danger');
        }
    } catch (error) {
        console.error('Error saving payment method:', error);
        showAlert('Error saving payment method', 'danger');
    }
}

// Delete method
function deleteMethod(id) {
    deleteCallback = async () => {
        try {
            const response = await fetch('api/accounts/payment-methods/delete.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ id: id })
            });

            const data = await response.json();

            if (data.success) {
                showAlert(data.message, 'success');
                showMethodsModal(); // Refresh list
                loadMethodsForFilters();
            } else {
                showAlert(data.message, 'danger');
            }
        } catch (error) {
            console.error('Error deleting payment method:', error);
            showAlert('Error deleting payment method', 'danger');
        }
    };

    document.getElementById('deleteMessage').textContent =
        'Are you sure you want to delete this payment method?';

    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

// Generate invoice
async function generateInvoice(transactionId) {
    try {
        // Load transaction details
        const response = await fetch(`api/accounts/transactions/get.php?id=${transactionId}`);
        const data = await response.json();

        if (!data.success) {
            showAlert('Failed to load transaction', 'danger');
            return;
        }

        const transaction = data.transaction;

        // Reset form
        document.getElementById('invoiceForm').reset();
        document.getElementById('invoiceTransactionId').value = transactionId;
        document.getElementById('invoiceId').value = '';

        // Set company info (from project name)
        document.getElementById('invoiceCompanyName').value = PROJECT_NAME;

        // Set invoice date to today
        document.getElementById('invoiceDate').valueAsDate = new Date();

        // Load contacts for customer selection
        await loadContactsForInvoice();

        // Load payment methods
        await loadMethodsForInvoice();

        // If transaction has contact, auto-fill
        if (transaction.contact_id) {
            document.getElementById('invoiceCustomerSelect').value = transaction.contact_id;
            fillCustomerDetails(transaction.contact_id);
        }

        // Add first line item with transaction details
        clearInvoiceItems();
        addInvoiceItem(transaction.description, transaction.amount);

        // Calculate totals
        calculateInvoiceTotals();

        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('invoiceModal'));
        modal.show();

    } catch (error) {
        console.error('Error generating invoice:', error);
        showAlert('Error loading invoice data', 'danger');
    }
}

// Load contacts for invoice
async function loadContactsForInvoice() {
    try {
        const response = await fetch(`api/accounts/contacts/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            allContacts = data.contacts; // Store for fillCustomerDetails
            const select = document.getElementById('invoiceCustomerSelect');
            select.innerHTML = '<option value="">Select or enter manually...</option>' +
                data.contacts.filter(c => c.type === 'customer' || c.type === 'both')
                    .map(c => `<option value="${c.id}">${escapeHtml(c.name)}</option>`).join('');

            // Add change listener
            select.addEventListener('change', function() {
                if (this.value) {
                    fillCustomerDetails(this.value);
                }
            });
        }
    } catch (error) {
        console.error('Error loading contacts for invoice:', error);
    }
}

// Load payment methods for invoice
async function loadMethodsForInvoice() {
    try {
        const response = await fetch(`api/accounts/payment-methods/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();

        if (data.success) {
            const select = document.getElementById('invoicePaymentMethod');
            select.innerHTML = '<option value="">Select method...</option>' +
                data.payment_methods.filter(m => m.type === 'income' || m.type === 'both')
                    .map(m => `<option value="${escapeHtml(m.name)}">${escapeHtml(m.name)}</option>`).join('');
        }
    } catch (error) {
        console.error('Error loading payment methods for invoice:', error);
    }
}

// Fill customer details
function fillCustomerDetails(contactId) {
    const contact = allContacts.find(c => c.id == contactId);
    if (contact) {
        document.getElementById('invoiceCustomerName').value = contact.name;
        document.getElementById('invoiceCustomerEmail').value = contact.email || '';
        document.getElementById('invoiceCustomerPhone').value = contact.phone || '';
        document.getElementById('invoiceCustomerAddress').value = contact.address || '';
    }
}

// Clear invoice items
function clearInvoiceItems() {
    document.getElementById('invoiceItems').innerHTML = '';
}

// Add invoice item
function addInvoiceItem(description = '', amount = '') {
    const itemsContainer = document.getElementById('invoiceItems');
    const itemIndex = itemsContainer.children.length;

    const itemDiv = document.createElement('div');
    itemDiv.className = 'invoice-item row mb-2';
    itemDiv.innerHTML = `
        <div class="col-md-7">
            <input type="text" class="form-control" placeholder="Service or item description"
                   value="${escapeHtml(description)}" data-field="service" required>
        </div>
        <div class="col-md-3">
            <input type="number" class="form-control" placeholder="Amount"
                   value="${amount}" step="0.01" min="0" data-field="amount" required>
        </div>
        <div class="col-md-2">
            <button type="button" class="btn btn-outline-danger w-100" onclick="removeInvoiceItem(this)">
                <i class="bi bi-trash"></i>
            </button>
        </div>
    `;

    // Add event listeners for auto-calculation
    itemDiv.querySelectorAll('input[data-field="amount"]').forEach(input => {
        input.addEventListener('input', calculateInvoiceTotals);
    });

    itemsContainer.appendChild(itemDiv);
}

// Remove invoice item
function removeInvoiceItem(button) {
    const itemsContainer = document.getElementById('invoiceItems');
    if (itemsContainer.children.length > 1) {
        button.closest('.invoice-item').remove();
        calculateInvoiceTotals();
    } else {
        showAlert('At least one item is required', 'warning');
    }
}

// Calculate invoice totals
function calculateInvoiceTotals() {
    const items = document.querySelectorAll('.invoice-item');
    let subtotal = 0;

    items.forEach(item => {
        const amount = parseFloat(item.querySelector('[data-field="amount"]').value) || 0;
        subtotal += amount;
    });

    const taxRate = parseFloat(document.getElementById('invoiceTaxRate').value) || 0;
    const taxAmount = (subtotal * taxRate) / 100;
    const total = subtotal + taxAmount;

    document.getElementById('invoiceSubtotal').textContent = '$' + formatNumber(subtotal);
    document.getElementById('invoiceTaxAmount').textContent = '$' + formatNumber(taxAmount);
    document.getElementById('invoiceTotal').textContent = '$' + formatNumber(total);
}

// Save invoice
async function saveInvoice() {
    // Validate form
    const form = document.getElementById('invoiceForm');
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }

    // Collect line items
    const items = [];
    document.querySelectorAll('.invoice-item').forEach(item => {
        const service = item.querySelector('[data-field="service"]').value;
        const amount = parseFloat(item.querySelector('[data-field="amount"]').value) || 0;
        if (service && amount > 0) {
            items.push({ service, amount });
        }
    });

    if (items.length === 0) {
        showAlert('Please add at least one item', 'warning');
        return;
    }

    const formData = {
        project_id: PROJECT_ID,
        transaction_id: document.getElementById('invoiceTransactionId').value,
        company_name: document.getElementById('invoiceCompanyName').value,
        company_email: document.getElementById('invoiceCompanyEmail').value,
        company_phone: document.getElementById('invoiceCompanyPhone').value,
        company_address: document.getElementById('invoiceCompanyAddress').value,
        customer_name: document.getElementById('invoiceCustomerName').value,
        customer_email: document.getElementById('invoiceCustomerEmail').value,
        customer_phone: document.getElementById('invoiceCustomerPhone').value,
        customer_address: document.getElementById('invoiceCustomerAddress').value,
        invoice_date: document.getElementById('invoiceDate').value,
        due_date: document.getElementById('invoiceDueDate').value,
        items: items,
        tax_rate: parseFloat(document.getElementById('invoiceTaxRate').value) || 0,
        payment_method: document.getElementById('invoicePaymentMethod').value,
        notes: document.getElementById('invoiceNotes').value
    };

    try {
        const response = await fetch('api/accounts/invoices/create.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(formData)
        });

        const data = await response.json();

        if (data.success) {
            showAlert('Invoice generated successfully!', 'success');
            bootstrap.Modal.getInstance(document.getElementById('invoiceModal')).hide();

            // Open invoice in new tab
            window.open(`api/accounts/invoices/generate-pdf.php?id=${data.invoice.id}`, '_blank');

            // Reload transactions to show invoice link
            loadTransactions();
            loadAccountStats();
        } else {
            showAlert(data.message, 'danger');
        }
    } catch (error) {
        console.error('Error saving invoice:', error);
        showAlert('Error generating invoice', 'danger');
    }
}

// Confirm delete
function confirmDelete() {
    if (deleteCallback) {
        deleteCallback();
        deleteCallback = null;
    }
    bootstrap.Modal.getInstance(document.getElementById('deleteModal')).hide();
}

// Utility functions
function formatNumber(num) {
    return parseFloat(num).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,');
}

function formatDate(dateString) {
    const date = new Date(dateString + 'T00:00:00');
    return date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function showAlert(message, type = 'info') {
    // Simple toast-style alert
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed top-0 start-50 translate-middle-x mt-3`;
    alertDiv.style.zIndex = '9999';
    alertDiv.style.minWidth = '300px';
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(alertDiv);

    // Auto-dismiss after 3 seconds
    setTimeout(() => {
        alertDiv.classList.remove('show');
        setTimeout(() => alertDiv.remove(), 150);
    }, 3000);
}
