/**
 * Dashboard Page JavaScript
 * Handles dashboard interactions, project CRUD, and statistics
 */

let currentEditProjectId = null;
let currentDeleteProjectId = null;

// Load dashboard data on page load
document.addEventListener('DOMContentLoaded', function() {
    loadDashboardStats();
    loadProjects();

    // Search input
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('input', debounce(function() {
            loadProjects();
        }, 500));
    }

    // Filter and sort
    const statusFilter = document.getElementById('statusFilter');
    const sortBy = document.getElementById('sortBy');

    if (statusFilter) {
        statusFilter.addEventListener('change', loadProjects);
    }

    if (sortBy) {
        sortBy.addEventListener('change', loadProjects);
    }

    // Save project button
    const saveProjectBtn = document.getElementById('saveProjectBtn');
    if (saveProjectBtn) {
        saveProjectBtn.addEventListener('click', saveProject);
    }

    // Confirm delete button
    const confirmDeleteBtn = document.getElementById('confirmDeleteBtn');
    if (confirmDeleteBtn) {
        confirmDeleteBtn.addEventListener('click', confirmDeleteProject);
    }

    // Reset modal on close
    const createProjectModal = document.getElementById('createProjectModal');
    if (createProjectModal) {
        createProjectModal.addEventListener('hidden.bs.modal', function() {
            resetProjectForm();
        });
    }
});

// Load dashboard statistics
async function loadDashboardStats() {
    try {
        const data = await apiCall('dashboard/stats.php');

        if (data.success) {
            const stats = data.stats;

            // Update stat cards
            document.getElementById('totalProjects').textContent = stats.projects.total;
            document.getElementById('workingProjects').textContent = stats.projects.working;
            document.getElementById('overdueTasks').textContent = stats.tasks.overdue || 0;
            document.getElementById('successProjects').textContent = stats.projects.success;
        }
    } catch (error) {
        console.error('Failed to load stats:', error);
    }
}

// Load projects
async function loadProjects() {
    const projectsList = document.getElementById('projectsList');
    showLoading(projectsList);

    try {
        // Get filter values
        const search = document.getElementById('searchInput')?.value || '';
        const status = document.getElementById('statusFilter')?.value || '';
        const sort = document.getElementById('sortBy')?.value || 'updated_at';

        // Build query string
        const params = new URLSearchParams();
        if (search) params.append('search', search);
        if (status) params.append('status', status);
        params.append('sort', sort);
        params.append('order', 'DESC');

        const data = await apiCall(`projects/list.php?${params.toString()}`);

        if (data.success) {
            if (data.projects.length === 0) {
                showEmptyState(projectsList, 'folder-x', 'No projects found. Create your first project to get started!');
            } else {
                renderProjects(data.projects);
            }
        }
    } catch (error) {
        console.error('Failed to load projects:', error);
        projectsList.innerHTML = '<div class="alert alert-danger">Failed to load projects. Please try again.</div>';
    }
}

// Render projects
function renderProjects(projects) {
    const projectsList = document.getElementById('projectsList');
    let html = '<div class="row g-3">';

    projects.forEach(project => {
        const percentage = project.completion_percentage || 0;
        const progressBarColor = percentage === 100 ? 'bg-success' : (percentage >= 50 ? 'bg-info' : 'bg-warning');

        html += `
            <div class="col-12 col-md-6 col-lg-4">
                <div class="card shadow-sm h-100 project-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <h5 class="card-title mb-0">
                                <a href="project.php?id=${project.id}" class="text-decoration-none text-dark">
                                    ${project.name}
                                </a>
                            </h5>
                            <div class="dropdown">
                                <button class="btn btn-sm btn-link text-secondary p-0" data-bs-toggle="dropdown">
                                    <i class="bi bi-three-dots-vertical fs-5"></i>
                                </button>
                                <ul class="dropdown-menu dropdown-menu-end">
                                    <li><a class="dropdown-item" href="project.php?id=${project.id}">
                                        <i class="bi bi-eye"></i> View
                                    </a></li>
                                    <li><a class="dropdown-item" href="#" onclick="editProject(${project.id}); return false;">
                                        <i class="bi bi-pencil"></i> Edit
                                    </a></li>
                                    <li><hr class="dropdown-divider"></li>
                                    <li><a class="dropdown-item text-danger" href="#" onclick="deleteProject(${project.id}); return false;">
                                        <i class="bi bi-trash"></i> Delete
                                    </a></li>
                                </ul>
                            </div>
                        </div>

                        <p class="card-text text-muted small">${project.description || 'No description'}</p>

                        <div class="mb-3">
                            ${getStatusBadge(project.status)}
                            <small class="text-muted ms-2">
                                <i class="bi bi-clock"></i> ${timeAgo(project.updated_at)}
                            </small>
                        </div>

                        <div class="mb-2">
                            <div class="d-flex justify-content-between align-items-center mb-1">
                                <small class="text-muted">Progress</small>
                                <small class="fw-semibold">${project.completed_tasks}/${project.total_tasks} tasks</small>
                            </div>
                            <div class="progress" style="height: 8px;">
                                <div class="progress-bar ${progressBarColor}" role="progressbar"
                                     style="width: ${percentage}%" aria-valuenow="${percentage}" aria-valuemin="0" aria-valuemax="100">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    });

    html += '</div>';
    projectsList.innerHTML = html;
}

// Edit project
async function editProject(projectId) {
    try {
        const data = await apiCall(`projects/get.php?id=${projectId}`);

        if (data.success) {
            const project = data.project;

            // Fill form
            document.getElementById('projectId').value = project.id;
            document.getElementById('projectName').value = project.name;
            document.getElementById('projectDescription').value = project.description || '';
            document.getElementById('projectStatus').value = project.status;

            // Update modal title
            document.getElementById('projectModalTitle').textContent = 'Edit Project';

            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('createProjectModal'));
            modal.show();

            currentEditProjectId = projectId;
        }
    } catch (error) {
        showToast('Failed to load project details', 'danger');
    }
}

// Save project (create or update)
async function saveProject() {
    const projectId = document.getElementById('projectId').value;
    const name = document.getElementById('projectName').value.trim();
    const description = document.getElementById('projectDescription').value.trim();
    const status = document.getElementById('projectStatus').value;

    if (!name) {
        showToast('Project name is required', 'warning');
        return;
    }

    const saveBtn = document.getElementById('saveProjectBtn');
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Saving...';

    try {
        const endpoint = projectId ? 'projects/update.php' : 'projects/create.php';
        const payload = { name, description, status };
        if (projectId) payload.id = projectId;

        const data = await apiCall(endpoint, {
            method: projectId ? 'PUT' : 'POST',
            body: JSON.stringify(payload)
        });

        if (data.success) {
            showToast(data.message);

            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('createProjectModal'));
            modal.hide();

            // Reload data
            loadDashboardStats();
            loadProjects();
        }
    } catch (error) {
        showToast(error.message || 'Failed to save project', 'danger');
    } finally {
        saveBtn.disabled = false;
        saveBtn.innerHTML = '<i class="bi bi-check-circle-fill me-1"></i>Save Project';
    }
}

// Delete project
function deleteProject(projectId) {
    currentDeleteProjectId = projectId;
    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

// Confirm delete project
async function confirmDeleteProject() {
    if (!currentDeleteProjectId) return;

    const confirmBtn = document.getElementById('confirmDeleteBtn');
    confirmBtn.disabled = true;
    confirmBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Deleting...';

    try {
        const data = await apiCall('projects/delete.php', {
            method: 'DELETE',
            body: JSON.stringify({ id: currentDeleteProjectId })
        });

        if (data.success) {
            showToast(data.message);

            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('deleteModal'));
            modal.hide();

            // Reload data
            loadDashboardStats();
            loadProjects();
        }
    } catch (error) {
        showToast(error.message || 'Failed to delete project', 'danger');
    } finally {
        confirmBtn.disabled = false;
        confirmBtn.innerHTML = '<i class="bi bi-trash-fill me-1"></i>Delete';
        currentDeleteProjectId = null;
    }
}

// Reset project form
function resetProjectForm() {
    document.getElementById('projectId').value = '';
    document.getElementById('projectName').value = '';
    document.getElementById('projectDescription').value = '';
    document.getElementById('projectStatus').value = 'working';
    document.getElementById('projectModalTitle').textContent = 'Create New Project';
    currentEditProjectId = null;
}

// Show overdue tasks modal
function showOverdueTasks() {
    const modal = new bootstrap.Modal(document.getElementById('overdueTasksModal'));
    modal.show();
    loadOverdueTasks();
}

// Load overdue tasks from API
async function loadOverdueTasks() {
    const contentDiv = document.getElementById('overdueTasksContent');
    showLoading(contentDiv);

    try {
        const data = await apiCall('dashboard/overdue-tasks.php');

        if (data.success) {
            if (data.overdue_tasks.length === 0) {
                showEmptyState(contentDiv, 'check-circle', 'No overdue tasks! You\'re all caught up.');
            } else {
                renderOverdueTasks(data.overdue_tasks);
            }
        }
    } catch (error) {
        console.error('Failed to load overdue tasks:', error);
        contentDiv.innerHTML = '<div class="alert alert-danger">Failed to load overdue tasks. Please try again.</div>';
    }
}

// Render overdue tasks grouped by project
function renderOverdueTasks(tasks) {
    const contentDiv = document.getElementById('overdueTasksContent');

    // Group tasks by project
    const projectGroups = {};
    tasks.forEach(task => {
        const projectId = task.project_id;
        if (!projectGroups[projectId]) {
            projectGroups[projectId] = {
                project_name: task.project_name,
                project_id: projectId,
                tasks: []
            };
        }
        projectGroups[projectId].tasks.push(task);
    });

    // Sort projects by number of overdue tasks (descending), then alphabetically
    const sortedProjects = Object.values(projectGroups).sort((a, b) => {
        if (b.tasks.length !== a.tasks.length) {
            return b.tasks.length - a.tasks.length;
        }
        return a.project_name.localeCompare(b.project_name);
    });

    let html = '';

    sortedProjects.forEach(project => {
        html += `
            <div class="card mb-3 shadow-sm">
                <div class="card-header bg-light">
                    <h6 class="mb-0">
                        <i class="bi bi-folder-fill me-2"></i>${project.project_name}
                        <span class="badge bg-danger ms-2">${project.tasks.length} overdue</span>
                    </h6>
                </div>
                <div class="list-group list-group-flush">
        `;

        // Sort tasks by days overdue (most overdue first)
        project.tasks.sort((a, b) => b.days_overdue - a.days_overdue);

        project.tasks.forEach(task => {
            const daysText = task.days_overdue === 1 ? '1 day' : `${task.days_overdue} days`;

            html += `
                <a href="project.php?id=${task.project_id}&task=${task.task_id}"
                   class="list-group-item list-group-item-action">
                    <div class="d-flex justify-content-between align-items-start mb-2">
                        <div class="flex-grow-1">
                            <h6 class="mb-1">${task.task_title}</h6>
                            <div>
                                ${getStatusBadge(task.task_status)}
                                ${getPriorityBadge(task.task_priority)}
                            </div>
                        </div>
                    </div>
                    <div class="d-flex justify-content-between align-items-center text-muted small">
                        <span>
                            <i class="bi bi-calendar-event"></i> Due: ${formatDate(task.task_due_date)}
                        </span>
                        <span class="text-danger fw-semibold">
                            <i class="bi bi-exclamation-triangle"></i> ${daysText} overdue
                        </span>
                    </div>
                </a>
            `;
        });

        html += `
                </div>
            </div>
        `;
    });

    contentDiv.innerHTML = html;
}
