// Load document notes
async function loadDocumentNotes() {
    try {
        const data = await apiCall(`projects/get.php?id=${PROJECT_ID}`);
        if (data.success && data.project) {
            document.getElementById('documentNotes').value = data.project.document_notes || '';
        }
    } catch (error) {
        console.error('Failed to load notes:', error);
    }
}

// Save document notes
async function saveNotes(e) {
    e.preventDefault();

    const notes = document.getElementById('documentNotes').value.trim();

    try {
        const response = await apiCall('projects/update-notes.php', {
            method: 'POST',
            body: JSON.stringify({
                project_id: PROJECT_ID,
                notes: notes
            })
        });

        if (response.success) {
            showToast(response.message, 'success');
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to save notes', 'danger');
    }
}

// Load URLs
async function loadUrls() {
    const urlsList = document.getElementById('urlsList');
    showLoading(urlsList);

    try {
        const data = await apiCall(`urls/list.php?project_id=${PROJECT_ID}`);

        if (data.success) {
            if (data.urls.length === 0) {
                showEmptyState(urlsList, 'link-45deg', 'No URLs added yet');
            } else {
                renderUrls(data.urls);
            }
        }
    } catch (error) {
        urlsList.innerHTML = '<div class="alert alert-danger">Failed to load URLs</div>';
    }
}

// Render URLs
function renderUrls(urls) {
    const urlsList = document.getElementById('urlsList');
    let html = '<div class="list-group">';

    urls.forEach(url => {
        const desc = url.description ? '<p class="mb-1 text-muted small">' + url.description + '</p>' : '';
        html += '<div class="list-group-item">';
        html += '  <div class="d-flex justify-content-between align-items-start">';
        html += '    <div class="flex-grow-1">';
        html += '      <h6 class="mb-1">';
        html += '        <a href="' + url.url + '" target="_blank" class="text-decoration-none">';
        html += '          <i class="bi bi-link-45deg"></i> ' + url.title;
        html += '          <i class="bi bi-box-arrow-up-right small ms-1"></i>';
        html += '        </a>';
        html += '      </h6>';
        html += desc;
        html += '      <small class="text-muted">';
        html += '        <i class="bi bi-calendar"></i> ' + timeAgo(url.created_at);
        html += '      </small>';
        html += '    </div>';
        html += '    <button class="btn btn-sm btn-danger" onclick="deleteUrl(' + url.id + ')">';
        html += '      <i class="bi bi-trash"></i>';
        html += '    </button>';
        html += '  </div>';
        html += '</div>';
    });

    html += '</div>';
    urlsList.innerHTML = html;
}

// Save URL
async function saveUrl() {
    const titleInput = document.getElementById('urlTitle');
    const urlInput = document.getElementById('urlAddress');
    const title = titleInput.value.trim();
    const urlAddress = urlInput.value.trim();
    const description = document.getElementById('urlDescription').value.trim();

    // Clear previous validation states
    titleInput.classList.remove('is-invalid');
    urlInput.classList.remove('is-invalid');

    // Validate
    let hasError = false;
    if (!title) {
        titleInput.classList.add('is-invalid');
        hasError = true;
    }
    if (!urlAddress) {
        urlInput.classList.add('is-invalid');
        hasError = true;
    }

    if (hasError) {
        showToast('Please fill in all required fields (Title and URL)', 'warning');
        return;
    }

    const btn = document.getElementById('saveUrlBtn');
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Adding...';

    try {
        const response = await apiCall('urls/create.php', {
            method: 'POST',
            body: JSON.stringify({
                project_id: PROJECT_ID,
                title: title,
                url: urlAddress,
                description: description
            })
        });

        if (response.success) {
            showToast(response.message, 'success');

            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('addUrlModal'));
            modal.hide();

            // Reset form
            document.getElementById('urlForm').reset();

            // Reload URLs and activity
            loadUrls();
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to add URL', 'danger');
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<i class="bi bi-plus-circle-fill me-1"></i>Add URL';
    }
}

// Delete URL
async function deleteUrl(urlId) {
    if (!confirm('Are you sure you want to delete this URL?')) return;

    try {
        const response = await apiCall('urls/delete.php', {
            method: 'DELETE',
            body: JSON.stringify({ id: urlId })
        });

        if (response.success) {
            showToast(response.message, 'success');
            loadUrls();
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to delete URL', 'danger');
    }
}
