/**
 * Project Page JavaScript
 * Handles project details, tasks, documents, and activity
 */

let currentEditTaskId = null;
let currentDeleteTaskId = null;

// Load project data on page load
document.addEventListener('DOMContentLoaded', function() {
    loadProjectDetails();
    loadTasks();
    loadDocuments();
    loadActivity();

    // Check for task URL parameter and auto-open edit modal
    const urlParams = new URLSearchParams(window.location.search);
    const taskIdParam = urlParams.get('task');

    if (taskIdParam) {
        const taskId = parseInt(taskIdParam);
        if (taskId && taskId > 0) {
            // Wait for tasks to load, then open the edit modal
            setTimeout(() => {
                openTaskFromUrl(taskId);
            }, 500); // Small delay to ensure tasks are loaded
        }
    }

    // Edit project button
    document.getElementById('editProjectBtn')?.addEventListener('click', editProject);

    // Delete project button
    document.getElementById('deleteProjectBtn')?.addEventListener('click', deleteProject);

    // Confirm delete project
    document.getElementById('confirmDeleteProjectBtn')?.addEventListener('click', confirmDeleteProject);

    // Save task button
    document.getElementById('saveTaskBtn')?.addEventListener('click', saveTask);

    // Confirm delete task
    document.getElementById('confirmDeleteTaskBtn')?.addEventListener('click', confirmDeleteTask);

    // Upload form
    document.getElementById('uploadForm')?.addEventListener('submit', uploadFiles);

    // Notes form
    document.getElementById('notesForm')?.addEventListener('submit', saveNotes);

    // URL form
    document.getElementById('saveUrlBtn')?.addEventListener('click', saveUrl);

    // Load notes and URLs
    loadDocumentNotes();
    loadUrls();

    // Reset task modal on close
    const createTaskModal = document.getElementById('createTaskModal');
    if (createTaskModal) {
        createTaskModal.addEventListener('hidden.bs.modal', resetTaskForm);
    }

    // Clear URL form validation on modal open
    const addUrlModal = document.getElementById('addUrlModal');
    if (addUrlModal) {
        addUrlModal.addEventListener('show.bs.modal', function() {
            document.getElementById('urlTitle')?.classList.remove('is-invalid');
            document.getElementById('urlAddress')?.classList.remove('is-invalid');
        });
    }
});

// Load project details
async function loadProjectDetails() {
    try {
        const data = await apiCall(`projects/get.php?id=${PROJECT_ID}`);

        if (data.success) {
            const project = data.project;

            document.getElementById('projectName').textContent = project.name;
            document.getElementById('projectDescription').textContent = project.description || 'No description';
            document.getElementById('projectStatusBadge').innerHTML = getStatusBadge(project.status);

            const percentage = project.completion_percentage || 0;
            const progressBar = document.getElementById('projectProgress');
            progressBar.style.width = percentage + '%';
            progressBar.textContent = percentage + '%';

            if (percentage === 100) {
                progressBar.classList.remove('progress-bar-animated');
                progressBar.classList.add('bg-success');
            }

            document.getElementById('projectProgressText').textContent =
                `${project.completed_tasks} of ${project.total_tasks} tasks completed`;
        }
    } catch (error) {
        showToast('Failed to load project details', 'danger');
    }
}

// Load tasks
async function loadTasks() {
    const tasksList = document.getElementById('tasksList');
    showLoading(tasksList);

    try {
        const data = await apiCall(`tasks/list.php?project_id=${PROJECT_ID}`);

        if (data.success) {
            if (data.tasks.length === 0) {
                showEmptyState(tasksList, 'list-task', 'No tasks yet. Add your first task to get started!');
            } else {
                renderTasks(data.tasks);
            }
        }
    } catch (error) {
        tasksList.innerHTML = '<div class="alert alert-danger">Failed to load tasks</div>';
    }
}

// Render tasks
function renderTasks(tasks) {
    const tasksList = document.getElementById('tasksList');
    let html = '';

    tasks.forEach(task => {
        const overdue = isOverdue(task.due_date, task.status);
        const overdueClass = overdue ? 'border-danger' : '';
        const overdueText = overdue ? '<span class="badge bg-danger ms-2"><i class="bi bi-exclamation-triangle"></i> Overdue</span>' : '';

        html += `
            <div class="card mb-3 shadow-sm task-card ${overdueClass}">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start mb-2">
                        <div class="flex-grow-1">
                            <h5 class="card-title mb-1">${task.title}</h5>
                            <div class="mb-2">
                                ${getStatusBadge(task.status)}
                                ${getPriorityBadge(task.priority)}
                                ${overdueText}
                            </div>
                        </div>
                        <div class="dropdown">
                            <button class="btn btn-sm btn-link text-secondary p-0" data-bs-toggle="dropdown">
                                <i class="bi bi-three-dots-vertical fs-5"></i>
                            </button>
                            <ul class="dropdown-menu dropdown-menu-end">
                                <li><a class="dropdown-item" href="#" onclick="editTask(${task.id}); return false;">
                                    <i class="bi bi-pencil"></i> Edit
                                </a></li>
                                <li><a class="dropdown-item" href="#" onclick="quickUpdateTaskStatus(${task.id}, '${task.status === 'completed' ? 'pending' : 'completed'}'); return false;">
                                    <i class="bi bi-${task.status === 'completed' ? 'arrow-counterclockwise' : 'check-circle'}"></i>
                                    ${task.status === 'completed' ? 'Mark Incomplete' : 'Mark Complete'}
                                </a></li>
                                <li><hr class="dropdown-divider"></li>
                                <li><a class="dropdown-item text-danger" href="#" onclick="deleteTask(${task.id}); return false;">
                                    <i class="bi bi-trash"></i> Delete
                                </a></li>
                            </ul>
                        </div>
                    </div>

                    ${task.description ? `<p class="text-muted small mb-2">${task.description}</p>` : ''}

                    ${task.notes ? `
                        <div class="alert alert-light mb-2">
                            <strong><i class="bi bi-sticky"></i> Notes:</strong><br>
                            ${task.notes}
                        </div>
                    ` : ''}

                    <div class="d-flex justify-content-between align-items-center text-muted small">
                        <span>${task.due_date ? '<i class="bi bi-calendar-event"></i> Due: ' + formatDate(task.due_date) : ''}</span>
                        <span><i class="bi bi-clock"></i> ${timeAgo(task.updated_at)}</span>
                    </div>
                </div>
            </div>
        `;
    });

    tasksList.innerHTML = html;
}

// Edit task
async function editTask(taskId) {
    try {
        const data = await apiCall(`tasks/get.php?id=${taskId}`);

        if (data.success) {
            const task = data.task;

            document.getElementById('taskId').value = task.id;
            document.getElementById('taskTitle').value = task.title;
            document.getElementById('taskDescription').value = task.description || '';
            document.getElementById('taskNotes').value = task.notes || '';
            document.getElementById('taskStatus').value = task.status;
            document.getElementById('taskPriority').value = task.priority;
            document.getElementById('taskDueDate').value = task.due_date || '';

            document.getElementById('taskModalTitle').textContent = 'Edit Task';

            const modal = new bootstrap.Modal(document.getElementById('createTaskModal'));
            modal.show();

            currentEditTaskId = taskId;
        }
    } catch (error) {
        showToast('Failed to load task details', 'danger');
    }
}

// Save task
async function saveTask() {
    const taskId = document.getElementById('taskId').value;
    const title = document.getElementById('taskTitle').value.trim();
    const description = document.getElementById('taskDescription').value.trim();
    const notes = document.getElementById('taskNotes').value.trim();
    const status = document.getElementById('taskStatus').value;
    const priority = document.getElementById('taskPriority').value;
    const dueDate = document.getElementById('taskDueDate').value;

    if (!title) {
        showToast('Task title is required', 'warning');
        return;
    }

    const saveBtn = document.getElementById('saveTaskBtn');
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Saving...';

    try {
        const endpoint = taskId ? 'tasks/update.php' : 'tasks/create.php';
        const payload = {
            title,
            description,
            notes,
            status,
            priority,
            due_date: dueDate || null
        };

        if (taskId) {
            payload.id = taskId;
        } else {
            payload.project_id = PROJECT_ID;
        }

        const data = await apiCall(endpoint, {
            method: taskId ? 'PUT' : 'POST',
            body: JSON.stringify(payload)
        });

        if (data.success) {
            showToast(data.message);

            const modal = bootstrap.Modal.getInstance(document.getElementById('createTaskModal'));
            modal.hide();

            loadProjectDetails();
            loadTasks();
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to save task', 'danger');
    } finally {
        saveBtn.disabled = false;
        saveBtn.innerHTML = '<i class="bi bi-check-circle-fill me-1"></i>Save Task';
    }
}

// Quick update task status
async function quickUpdateTaskStatus(taskId, status) {
    try {
        const data = await apiCall('tasks/update.php', {
            method: 'PUT',
            body: JSON.stringify({ id: taskId, status })
        });

        if (data.success) {
            showToast('Task updated');
            loadProjectDetails();
            loadTasks();
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to update task', 'danger');
    }
}

// Delete task
function deleteTask(taskId) {
    currentDeleteTaskId = taskId;
    const modal = new bootstrap.Modal(document.getElementById('deleteTaskModal'));
    modal.show();
}

// Confirm delete task
async function confirmDeleteTask() {
    if (!currentDeleteTaskId) return;

    const confirmBtn = document.getElementById('confirmDeleteTaskBtn');
    confirmBtn.disabled = true;
    confirmBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Deleting...';

    try {
        const data = await apiCall('tasks/delete.php', {
            method: 'DELETE',
            body: JSON.stringify({ id: currentDeleteTaskId })
        });

        if (data.success) {
            showToast(data.message);

            const modal = bootstrap.Modal.getInstance(document.getElementById('deleteTaskModal'));
            modal.hide();

            loadProjectDetails();
            loadTasks();
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to delete task', 'danger');
    } finally {
        confirmBtn.disabled = false;
        confirmBtn.innerHTML = 'Delete';
        currentDeleteTaskId = null;
    }
}

// Reset task form
function resetTaskForm() {
    document.getElementById('taskId').value = '';
    document.getElementById('taskTitle').value = '';
    document.getElementById('taskDescription').value = '';
    document.getElementById('taskNotes').value = '';
    document.getElementById('taskStatus').value = 'pending';
    document.getElementById('taskPriority').value = 'medium';
    document.getElementById('taskDueDate').value = '';
    document.getElementById('taskModalTitle').textContent = 'Create New Task';
    currentEditTaskId = null;
}

// Upload files
async function uploadFiles(e) {
    e.preventDefault();

    const fileInput = document.getElementById('fileInput');
    const files = fileInput.files;

    if (files.length === 0) {
        showToast('Please select files to upload', 'warning');
        return;
    }

    const formData = new FormData();
    formData.append('project_id', PROJECT_ID);

    for (let i = 0; i < files.length; i++) {
        formData.append('files[]', files[i]);
    }

    try {
        const response = await fetch(`${API_BASE}/documents/upload.php`, {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            showToast(data.message);
            fileInput.value = '';
            loadDocuments();
            loadActivity();
        } else {
            showToast(data.message || 'Upload failed', 'danger');
        }
    } catch (error) {
        showToast('Failed to upload files', 'danger');
    }
}

// Load documents
async function loadDocuments() {
    const documentsList = document.getElementById('documentsList');
    showLoading(documentsList);

    try {
        const data = await apiCall(`documents/list.php?project_id=${PROJECT_ID}`);

        if (data.success) {
            if (data.documents.length === 0) {
                showEmptyState(documentsList, 'file-earmark-x', 'No documents uploaded yet');
            } else {
                renderDocuments(data.documents);
            }
        }
    } catch (error) {
        documentsList.innerHTML = '<div class="alert alert-danger">Failed to load documents</div>';
    }
}

// Render documents
function renderDocuments(documents) {
    const documentsList = document.getElementById('documentsList');
    let html = '<div class="list-group">';

    documents.forEach(doc => {
        html += `
            <div class="list-group-item d-flex justify-content-between align-items-center">
                <div class="flex-grow-1">
                    <h6 class="mb-1"><i class="bi bi-file-earmark"></i> ${doc.original_filename}</h6>
                    <small class="text-muted">
                        ${doc.file_size_formatted} • Uploaded ${timeAgo(doc.uploaded_at)}
                    </small>
                </div>
                <div class="btn-group">
                    <a href="${API_BASE}/documents/download.php?id=${doc.id}" class="btn btn-sm btn-primary">
                        <i class="bi bi-download"></i>
                    </a>
                    <button class="btn btn-sm btn-danger" onclick="deleteDocument(${doc.id})">
                        <i class="bi bi-trash"></i>
                    </button>
                </div>
            </div>
        `;
    });

    html += '</div>';
    documentsList.innerHTML = html;
}

// Delete document
async function deleteDocument(documentId) {
    if (!confirm('Are you sure you want to delete this document?')) return;

    try {
        const data = await apiCall('documents/delete.php', {
            method: 'DELETE',
            body: JSON.stringify({ id: documentId })
        });

        if (data.success) {
            showToast(data.message);
            loadDocuments();
            loadActivity();
        }
    } catch (error) {
        showToast(error.message || 'Failed to delete document', 'danger');
    }
}

// Load activity
async function loadActivity() {
    const activityList = document.getElementById('activityList');
    showLoading(activityList);

    try {
        const data = await apiCall(`activity/list.php?project_id=${PROJECT_ID}&limit=50`);

        if (data.success) {
            if (data.activities.length === 0) {
                showEmptyState(activityList, 'clock-history', 'No activity yet');
            } else {
                renderActivity(data.activities);
            }
        }
    } catch (error) {
        activityList.innerHTML = '<div class="alert alert-danger">Failed to load activity</div>';
    }
}

// Render activity
function renderActivity(activities) {
    const activityList = document.getElementById('activityList');
    let html = '<div class="timeline">';

    activities.forEach(activity => {
        const icon = getActivityIcon(activity.action);

        html += `
            <div class="timeline-item">
                <div class="timeline-marker ${icon.color}">
                    <i class="bi bi-${icon.icon}"></i>
                </div>
                <div class="timeline-content">
                    <p class="mb-1">${activity.description}</p>
                    <small class="text-muted">${timeAgo(activity.created_at)}</small>
                </div>
            </div>
        `;
    });

    html += '</div>';
    activityList.innerHTML = html;
}

// Get activity icon
function getActivityIcon(action) {
    const icons = {
        create_project: { icon: 'folder-plus', color: 'bg-primary' },
        update_project: { icon: 'pencil', color: 'bg-info' },
        delete_project: { icon: 'trash', color: 'bg-danger' },
        create_task: { icon: 'plus-circle', color: 'bg-success' },
        update_task: { icon: 'check-circle', color: 'bg-info' },
        delete_task: { icon: 'x-circle', color: 'bg-danger' },
        upload_document: { icon: 'cloud-upload', color: 'bg-primary' },
        delete_document: { icon: 'file-x', color: 'bg-danger' },
        update_document_notes: { icon: 'sticky', color: 'bg-info' },
        add_url: { icon: 'link-45deg', color: 'bg-success' },
        delete_url: { icon: 'link-45deg', color: 'bg-danger' }
    };
    return icons[action] || { icon: 'circle', color: 'bg-secondary' };
}

// Edit project (redirect to dashboard with edit modal)
function editProject() {
    // For simplicity, we'll just open an edit modal here
    showToast('Edit project feature - redirect to dashboard or implement inline editing', 'warning');
}

// Delete project
function deleteProject() {
    const modal = new bootstrap.Modal(document.getElementById('deleteProjectModal'));
    modal.show();
}

// Confirm delete project
async function confirmDeleteProject() {
    const confirmBtn = document.getElementById('confirmDeleteProjectBtn');
    confirmBtn.disabled = true;
    confirmBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Deleting...';

    try {
        const data = await apiCall('projects/delete.php', {
            method: 'DELETE',
            body: JSON.stringify({ id: PROJECT_ID })
        });

        if (data.success) {
            showToast(data.message);
            setTimeout(() => {
                window.location.href = 'dashboard.php';
            }, 1000);
        }
    } catch (error) {
        showToast(error.message || 'Failed to delete project', 'danger');
        confirmBtn.disabled = false;
        confirmBtn.innerHTML = 'Delete';
    }
}

// Open task edit modal from URL parameter
async function openTaskFromUrl(taskId) {
    try {
        // Verify task belongs to this project
        const data = await apiCall(`tasks/get.php?id=${taskId}`);

        if (data.success && data.task) {
            // Check if task belongs to current project
            if (data.task.project_id === PROJECT_ID) {
                editTask(taskId);

                // Remove task parameter from URL without reload
                const url = new URL(window.location);
                url.searchParams.delete('task');
                window.history.replaceState({}, '', url);
            } else {
                showToast('Task not found in this project', 'warning');
            }
        } else {
            showToast('Task not found', 'warning');
        }
    } catch (error) {
        console.error('Failed to open task from URL:', error);
        showToast('Could not open task', 'danger');
    }
}
