<?php
/**
 * Utility Functions
 * Helper functions for file uploads, activity logging, etc.
 */

// Config should already be loaded, but include it if not
if (!defined('DB_HOST')) {
    require_once __DIR__ . '/config.php';
}

require_once __DIR__ . '/db.php';

/**
 * Sanitize output to prevent XSS
 */
function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Log activity
 */
function logActivity($userId, $action, $description, $projectId = null, $taskId = null) {
    $db = getDB();

    try {
        $stmt = $db->prepare("
            INSERT INTO activity_logs (user_id, project_id, task_id, action, description)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$userId, $projectId, $taskId, $action, $description]);
        return true;
    } catch (PDOException $e) {
        error_log("Activity Log Error: " . $e->getMessage());
        return false;
    }
}

/**
 * Handle file upload
 */
function handleFileUpload($file, $projectId) {
    // Validate file
    if (!isset($file['error']) || is_array($file['error'])) {
        return ['success' => false, 'message' => 'Invalid file upload'];
    }

    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'message' => 'File upload error: ' . $file['error']];
    }

    // Check file size
    if ($file['size'] > MAX_FILE_SIZE) {
        return ['success' => false, 'message' => 'File size exceeds maximum allowed (' . formatFileSize(MAX_FILE_SIZE) . ')'];
    }

    // Get file extension
    $originalFilename = basename($file['name']);
    $fileExtension = strtolower(pathinfo($originalFilename, PATHINFO_EXTENSION));

    // Validate file type
    if (!in_array($fileExtension, ALLOWED_FILE_TYPES)) {
        return ['success' => false, 'message' => 'File type not allowed. Allowed types: ' . implode(', ', ALLOWED_FILE_TYPES)];
    }

    // Generate unique filename
    $uniqueFilename = uniqid() . '_' . time() . '.' . $fileExtension;
    $uploadPath = UPLOAD_DIR . $uniqueFilename;

    // Create upload directory if it doesn't exist
    if (!is_dir(UPLOAD_DIR)) {
        mkdir(UPLOAD_DIR, 0755, true);
    }

    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
        return ['success' => false, 'message' => 'Failed to move uploaded file'];
    }

    // Save to database
    $db = getDB();
    try {
        $stmt = $db->prepare("
            INSERT INTO documents (project_id, filename, original_filename, file_path, file_size)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $projectId,
            $uniqueFilename,
            $originalFilename,
            $uploadPath,
            $file['size']
        ]);

        return [
            'success' => true,
            'message' => 'File uploaded successfully',
            'document_id' => $db->lastInsertId(),
            'filename' => $uniqueFilename
        ];
    } catch (PDOException $e) {
        // Delete file if database insert fails
        if (file_exists($uploadPath)) {
            unlink($uploadPath);
        }
        error_log("Document Upload Error: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to save document information'];
    }
}

/**
 * Delete file
 */
function deleteFile($documentId, $userId) {
    $db = getDB();

    // Get document info
    $stmt = $db->prepare("
        SELECT d.*, p.user_id
        FROM documents d
        JOIN projects p ON d.project_id = p.id
        WHERE d.id = ?
    ");
    $stmt->execute([$documentId]);
    $document = $stmt->fetch();

    if (!$document) {
        return ['success' => false, 'message' => 'Document not found'];
    }

    // Check permission
    if ($document['user_id'] != $userId) {
        return ['success' => false, 'message' => 'Unauthorized'];
    }

    // Delete physical file
    if (file_exists($document['file_path'])) {
        unlink($document['file_path']);
    }

    // Delete from database
    try {
        $stmt = $db->prepare("DELETE FROM documents WHERE id = ?");
        $stmt->execute([$documentId]);

        return ['success' => true, 'message' => 'Document deleted successfully'];
    } catch (PDOException $e) {
        error_log("Document Delete Error: " . $e->getMessage());
        return ['success' => false, 'message' => 'Failed to delete document'];
    }
}

/**
 * Format file size
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= (1 << (10 * $pow));

    return round($bytes, 2) . ' ' . $units[$pow];
}

/**
 * Format date for display
 */
function formatDate($date, $format = 'M d, Y') {
    return date($format, strtotime($date));
}

/**
 * Format datetime for display
 */
function formatDateTime($datetime, $format = 'M d, Y g:i A') {
    return date($format, strtotime($datetime));
}

/**
 * Get time ago string
 */
function timeAgo($datetime) {
    $timestamp = strtotime($datetime);
    $diff = time() - $timestamp;

    if ($diff < 60) {
        return 'just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' minute' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 604800) {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    } else {
        return formatDate($datetime);
    }
}

/**
 * Check if project belongs to user
 */
function userOwnsProject($projectId, $userId) {
    $db = getDB();
    $stmt = $db->prepare("SELECT id FROM projects WHERE id = ? AND user_id = ?");
    $stmt->execute([$projectId, $userId]);
    return $stmt->fetch() !== false;
}

/**
 * Send JSON response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Validate CSRF token
 */
function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Generate CSRF token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}
