<?php
require_once 'includes/config.php';
require_once 'includes/auth.php';
require_once 'includes/functions.php';
require_once 'includes/db.php';

requireAuth();
$user = getCurrentUser();

// Get full user details from database
$db = getDB();
$stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user['id']]);
$userDetails = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Profile Settings - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.2/font/bootstrap-icons.css">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <!-- Top Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary shadow-sm sticky-top">
        <div class="container-fluid">
            <a class="navbar-brand fw-bold" href="dashboard.php">
                <i class="bi bi-arrow-left-circle me-2"></i>Back to Dashboard
            </a>
            <div class="d-flex align-items-center">
                <span class="navbar-text text-white me-3">
                    <i class="bi bi-person-circle me-1"></i><?php echo e($user['username']); ?>
                </span>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0"><i class="bi bi-person-circle"></i> Account Settings</h4>
                    </div>
                    <div class="card-body">
                        <div class="text-center mb-4">
                            <div class="avatar-placeholder mb-3">
                                <i class="bi bi-person-circle text-primary" style="font-size: 5rem;"></i>
                            </div>
                            <h5><?php echo e($userDetails['name'] ?? $userDetails['username']); ?></h5>
                            <p class="text-muted"><?php echo e($userDetails['email']); ?></p>
                            <span class="badge bg-success">
                                <i class="bi bi-calendar-check"></i> Member since <?php echo date('M Y', strtotime($userDetails['created_at'])); ?>
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Update Profile Form -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-pencil-square"></i> Update Profile Information</h5>
                    </div>
                    <div class="card-body">
                        <form id="profileForm">
                            <div class="mb-3">
                                <label for="name" class="form-label fw-semibold">Full Name</label>
                                <input type="text" class="form-control form-control-lg" id="name"
                                       value="<?php echo e($userDetails['name'] ?? ''); ?>" placeholder="Enter your full name">
                                <small class="text-muted">Your display name (optional)</small>
                            </div>
                            <div class="mb-3">
                                <label for="username" class="form-label fw-semibold">Username</label>
                                <input type="text" class="form-control form-control-lg" id="username"
                                       value="<?php echo e($userDetails['username']); ?>" required>
                                <small class="text-muted">Used for login</small>
                            </div>
                            <div class="mb-3">
                                <label for="email" class="form-label fw-semibold">Email Address</label>
                                <input type="email" class="form-control form-control-lg" id="email"
                                       value="<?php echo e($userDetails['email']); ?>" required>
                                <small class="text-muted">Used for login and notifications</small>
                            </div>
                            <button type="submit" class="btn btn-primary btn-lg w-100" id="updateProfileBtn">
                                <i class="bi bi-check-circle-fill me-2"></i>Update Profile
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Change Password Form -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-shield-lock"></i> Change Password</h5>
                    </div>
                    <div class="card-body">
                        <form id="passwordForm">
                            <div class="mb-3">
                                <label for="currentPassword" class="form-label fw-semibold">Current Password</label>
                                <div class="input-group input-group-lg">
                                    <input type="password" class="form-control" id="currentPassword" required>
                                    <button class="btn btn-outline-secondary" type="button" id="toggleCurrent">
                                        <i class="bi bi-eye-fill"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label for="newPassword" class="form-label fw-semibold">New Password</label>
                                <div class="input-group input-group-lg">
                                    <input type="password" class="form-control" id="newPassword" required minlength="6">
                                    <button class="btn btn-outline-secondary" type="button" id="toggleNew">
                                        <i class="bi bi-eye-fill"></i>
                                    </button>
                                </div>
                                <small class="text-muted">Minimum 6 characters</small>
                            </div>
                            <div class="mb-3">
                                <label for="confirmPassword" class="form-label fw-semibold">Confirm New Password</label>
                                <div class="input-group input-group-lg">
                                    <input type="password" class="form-control" id="confirmPassword" required minlength="6">
                                    <button class="btn btn-outline-secondary" type="button" id="toggleConfirm">
                                        <i class="bi bi-eye-fill"></i>
                                    </button>
                                </div>
                            </div>
                            <button type="submit" class="btn btn-warning btn-lg w-100" id="updatePasswordBtn">
                                <i class="bi bi-shield-check me-2"></i>Change Password
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Danger Zone -->
                <div class="card shadow-sm border-danger mb-4">
                    <div class="card-header bg-danger text-white">
                        <h5 class="mb-0"><i class="bi bi-exclamation-triangle-fill"></i> Danger Zone</h5>
                    </div>
                    <div class="card-body">
                        <p class="text-muted mb-3">Once you delete your account, there is no going back. All your projects, tasks, and documents will be permanently deleted.</p>
                        <button class="btn btn-outline-danger" onclick="alert('Account deletion feature coming soon!')">
                            <i class="bi bi-trash-fill me-2"></i>Delete Account
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Mobile Bottom Navigation -->
    <nav class="mobile-bottom-nav d-md-none">
        <a href="dashboard.php" class="nav-item">
            <i class="bi bi-house-fill"></i>
            <span>Dashboard</span>
        </a>
        <a href="profile.php" class="nav-item active">
            <i class="bi bi-person-circle"></i>
            <span>Profile</span>
        </a>
        <a href="logout.php" class="nav-item">
            <i class="bi bi-box-arrow-right"></i>
            <span>Logout</span>
        </a>
    </nav>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/app.js"></script>
    <script>
        // Update Profile Form
        document.getElementById('profileForm').addEventListener('submit', async function(e) {
            e.preventDefault();

            const name = document.getElementById('name').value.trim();
            const username = document.getElementById('username').value.trim();
            const email = document.getElementById('email').value.trim();

            const btn = document.getElementById('updateProfileBtn');
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Updating...';

            try {
                const response = await fetch('api/user/update-profile.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ name, username, email })
                });

                const data = await response.json();

                if (data.success) {
                    showToast(data.message, 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showToast(data.message, 'danger');
                }
            } catch (error) {
                showToast('Failed to update profile', 'danger');
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="bi bi-check-circle-fill me-2"></i>Update Profile';
            }
        });

        // Change Password Form
        document.getElementById('passwordForm').addEventListener('submit', async function(e) {
            e.preventDefault();

            const currentPassword = document.getElementById('currentPassword').value;
            const newPassword = document.getElementById('newPassword').value;
            const confirmPassword = document.getElementById('confirmPassword').value;

            if (newPassword !== confirmPassword) {
                showToast('New passwords do not match', 'warning');
                return;
            }

            if (newPassword.length < 6) {
                showToast('Password must be at least 6 characters', 'warning');
                return;
            }

            const btn = document.getElementById('updatePasswordBtn');
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Updating...';

            try {
                const response = await fetch('api/user/update-password.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ currentPassword, newPassword })
                });

                const data = await response.json();

                if (data.success) {
                    showToast(data.message, 'success');
                    document.getElementById('passwordForm').reset();
                } else {
                    showToast(data.message, 'danger');
                }
            } catch (error) {
                showToast('Failed to update password', 'danger');
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="bi bi-shield-check me-2"></i>Change Password';
            }
        });

        // Toggle password visibility
        ['toggleCurrent', 'toggleNew', 'toggleConfirm'].forEach(id => {
            document.getElementById(id)?.addEventListener('click', function() {
                const input = this.previousElementSibling;
                const icon = this.querySelector('i');
                if (input.type === 'password') {
                    input.type = 'text';
                    icon.classList.replace('bi-eye-fill', 'bi-eye-slash-fill');
                } else {
                    input.type = 'password';
                    icon.classList.replace('bi-eye-slash-fill', 'bi-eye-fill');
                }
            });
        });
    </script>
</body>
</html>
