<?php
/**
 * Document Features Setup
 * Run this once to add document_notes column and project_urls table
 */

require_once 'includes/config.php';
require_once 'includes/db.php';

$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['setup'])) {
    try {
        $db = getDB();
        $errors = [];
        $success = [];

        // Add document_notes column
        try {
            $stmt = $db->query("SHOW COLUMNS FROM projects LIKE 'document_notes'");
            if ($stmt->rowCount() > 0) {
                $success[] = "document_notes column already exists";
            } else {
                $db->exec("ALTER TABLE projects ADD COLUMN document_notes TEXT NULL AFTER description");
                $success[] = "document_notes column added successfully";
            }
        } catch (Exception $e) {
            $errors[] = "Error adding document_notes: " . $e->getMessage();
        }

        // Create project_urls table
        try {
            $stmt = $db->query("SHOW TABLES LIKE 'project_urls'");
            if ($stmt->rowCount() > 0) {
                $success[] = "project_urls table already exists";
            } else {
                $db->exec("
                    CREATE TABLE project_urls (
                        id INT PRIMARY KEY AUTO_INCREMENT,
                        project_id INT NOT NULL,
                        title VARCHAR(200) NOT NULL,
                        url VARCHAR(500) NOT NULL,
                        description TEXT,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
                        INDEX idx_project_id (project_id)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                ");
                $success[] = "project_urls table created successfully";
            }
        } catch (Exception $e) {
            $errors[] = "Error creating project_urls table: " . $e->getMessage();
        }

        if (empty($errors)) {
            $message = implode("<br>", $success);
            $messageType = "success";
        } else {
            $message = implode("<br>", array_merge($success, $errors));
            $messageType = count($success) > 0 ? "warning" : "danger";
        }
    } catch (Exception $e) {
        $message = "Error: " . $e->getMessage();
        $messageType = "danger";
    }
}

// Check current status
try {
    $db = getDB();
    $stmt = $db->query("SHOW COLUMNS FROM projects LIKE 'document_notes'");
    $notesColumnExists = $stmt->rowCount() > 0;

    $stmt = $db->query("SHOW TABLES LIKE 'project_urls'");
    $urlsTableExists = $stmt->rowCount() > 0;

    $setupComplete = $notesColumnExists && $urlsTableExists;
} catch (Exception $e) {
    $notesColumnExists = false;
    $urlsTableExists = false;
    $setupComplete = false;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Document Features Setup - <?php echo APP_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.2/font/bootstrap-icons.css">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-md-8 col-lg-6">
                <div class="card shadow-lg">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0"><i class="bi bi-gear-fill"></i> Document Features Setup</h4>
                    </div>
                    <div class="card-body">
                        <h5>Add Notes & URLs to Documents Tab</h5>
                        <p class="text-muted">This will add document notes and URLs features to your projects.</p>

                        <?php if ($message): ?>
                            <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                                <?php if ($messageType === 'success'): ?>
                                    <i class="bi bi-check-circle-fill"></i>
                                <?php elseif ($messageType === 'warning'): ?>
                                    <i class="bi bi-exclamation-triangle-fill"></i>
                                <?php else: ?>
                                    <i class="bi bi-x-circle-fill"></i>
                                <?php endif; ?>
                                <?php echo $message; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>

                        <div class="alert alert-info">
                            <strong><i class="bi bi-info-circle"></i> Current Status:</strong><br>
                            document_notes column: <?php echo $notesColumnExists ? '<span class="badge bg-success">Exists</span>' : '<span class="badge bg-warning">Not found</span>'; ?><br>
                            project_urls table: <?php echo $urlsTableExists ? '<span class="badge bg-success">Exists</span>' : '<span class="badge bg-warning">Not found</span>'; ?>
                        </div>

                        <?php if (!$setupComplete): ?>
                            <form method="POST">
                                <div class="alert alert-warning">
                                    <strong>⚠️ This setup will:</strong>
                                    <ul class="mb-0">
                                        <li>Add document_notes column to projects table</li>
                                        <li>Create project_urls table for storing URLs</li>
                                        <li>Enable Notes and URLs in Documents tab</li>
                                    </ul>
                                </div>

                                <button type="submit" name="setup" class="btn btn-primary btn-lg w-100">
                                    <i class="bi bi-plus-circle-fill"></i> Run Setup
                                </button>
                            </form>
                        <?php else: ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle-fill"></i> Setup complete! Document features are ready to use.
                            </div>

                            <a href="dashboard.php" class="btn btn-primary btn-lg w-100 mb-2">
                                <i class="bi bi-house-fill"></i> Go to Dashboard
                            </a>
                            <a href="test-config.php" class="btn btn-secondary btn-lg w-100">
                                <i class="bi bi-gear"></i> Run Diagnostics
                            </a>
                        <?php endif; ?>

                        <hr class="my-4">

                        <h6>What's New:</h6>
                        <ul class="small">
                            <li><strong>Notes:</strong> Add notes/instructions for documents</li>
                            <li><strong>URLs:</strong> Add useful links with descriptions</li>
                            <li><strong>Files:</strong> Upload documents (already working)</li>
                        </ul>

                        <p class="small text-muted">
                            The Documents tab will show these sections in order: Notes, URLs, then Files.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
