<?php
session_start();
require_once 'includes/config.php';
require_once 'includes/db.php';
require_once 'includes/auth.php';

// Check if user is logged in
if (!isLoggedIn()) {
    ?>
    <h2>Please log in first</h2>
    <a href='login.php'>Go to Login</a>
    <?php
    exit;
}

$userId = getCurrentUserId();
$db = getDB();

// Get first project for testing
$stmt = $db->prepare("SELECT id, name FROM projects WHERE user_id = ? LIMIT 1");
$stmt->execute([$userId]);
$project = $stmt->fetch();

if (!$project) {
    ?>
    <h2>API Endpoints Test</h2>
    <p style='color: red;'>❌ No projects found. Please create a project first.</p>
    <a href='dashboard.php'>Go to Dashboard</a>
    <?php
    exit;
}

$projectId = $project['id'];
?>

<h2>API Endpoints Test</h2>
<p>Testing as User ID: <?php echo $userId; ?></p>
<h3>Testing with Project: <?php echo $project['name']; ?> (ID: <?php echo $projectId; ?>)</h3>
<hr>

<!-- Test 1: Update Notes API -->
<h4>Test 1: Update Notes API</h4>
<form method='post' action='api/projects/update-notes.php' id='notesTestForm'>
    <input type='hidden' name='test_mode' value='1'>
    <textarea name='notes' rows='3' style='width: 100%; padding: 10px;' placeholder='Enter test notes here'>Test notes from diagnostic page</textarea>
    <br><br>
    <button type='button' onclick='testNotesAPI()' style='padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer;'>Test Save Notes</button>
</form>
<div id='notesResult' style='margin-top: 10px;'></div>

<hr>

<!-- Test 2: Add URL API -->
<h4>Test 2: Add URL API</h4>
<form id='urlTestForm'>
    <label>Title:</label><br>
    <input type='text' id='urlTitle' value='Test URL Link' style='width: 100%; padding: 8px; margin-bottom: 10px;'><br>

    <label>URL:</label><br>
    <input type='url' id='urlAddress' value='https://google.com' style='width: 100%; padding: 8px; margin-bottom: 10px;'><br>

    <label>Description:</label><br>
    <textarea id='urlDescription' rows='2' style='width: 100%; padding: 8px; margin-bottom: 10px;'>Testing URL creation</textarea><br>

    <button type='button' onclick='testUrlAPI()' style='padding: 10px 20px; background: #28a745; color: white; border: none; border-radius: 5px; cursor: pointer;'>Test Add URL</button>
</form>
<div id='urlResult' style='margin-top: 10px;'></div>

<hr>

<!-- Test 3: List URLs API -->
<h4>Test 3: List URLs API</h4>
<button onclick='testListUrls()' style='padding: 10px 20px; background: #17a2b8; color: white; border: none; border-radius: 5px; cursor: pointer;'>Test List URLs</button>
<div id='listResult' style='margin-top: 10px;'></div>

<hr>
<p><a href='dashboard.php'>← Back to Dashboard</a> | <a href='project.php?id=<?php echo $projectId; ?>'>View This Project</a></p>

<script>
const PROJECT_ID = <?php echo $projectId; ?>;
const API_BASE = 'api';

function showResult(elementId, message, isSuccess) {
    const el = document.getElementById(elementId);
    const bgColor = isSuccess ? '#d4edda' : '#f8d7da';
    const textColor = isSuccess ? '#155724' : '#721c24';
    const borderColor = isSuccess ? '#c3e6cb' : '#f5c6cb';

    const div = document.createElement('div');
    div.style.padding = '15px';
    div.style.margin = '10px 0';
    div.style.borderRadius = '5px';
    div.style.background = bgColor;
    div.style.color = textColor;
    div.style.border = '1px solid ' + borderColor;
    div.innerHTML = message;

    el.innerHTML = '';
    el.appendChild(div);
}

async function testNotesAPI() {
    const notes = document.querySelector('textarea[name="notes"]').value;

    try {
        const response = await fetch(`${API_BASE}/projects/update-notes.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                project_id: PROJECT_ID,
                notes: notes
            })
        });

        const data = await response.json();
        console.log('Notes API Response:', data);

        if (data.success) {
            showResult('notesResult', '✅ SUCCESS: ' + data.message, true);
        } else {
            showResult('notesResult', '❌ ERROR: ' + data.message, false);
        }
    } catch (error) {
        console.error('Notes API Error:', error);
        showResult('notesResult', '❌ EXCEPTION: ' + error.message, false);
    }
}

async function testUrlAPI() {
    const title = document.getElementById('urlTitle').value;
    const url = document.getElementById('urlAddress').value;
    const description = document.getElementById('urlDescription').value;

    try {
        const response = await fetch(`${API_BASE}/urls/create.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                project_id: PROJECT_ID,
                title: title,
                url: url,
                description: description
            })
        });

        const data = await response.json();
        console.log('URL API Response:', data);

        if (data.success) {
            showResult('urlResult', '✅ SUCCESS: ' + data.message + ' (ID: ' + data.url_id + ')', true);
        } else {
            showResult('urlResult', '❌ ERROR: ' + data.message, false);
        }
    } catch (error) {
        console.error('URL API Error:', error);
        showResult('urlResult', '❌ EXCEPTION: ' + error.message, false);
    }
}

async function testListUrls() {
    try {
        const response = await fetch(`${API_BASE}/urls/list.php?project_id=${PROJECT_ID}`);
        const data = await response.json();
        console.log('List URLs API Response:', data);

        if (data.success) {
            let html = '✅ SUCCESS: Found ' + data.urls.length + ' URLs<br><br>';

            if (data.urls.length > 0) {
                html += '<table style="width: 100%; border-collapse: collapse;">';
                html += '<tr style="background: #f0f0f0;"><th style="padding: 8px; border: 1px solid #ddd;">ID</th><th style="padding: 8px; border: 1px solid #ddd;">Title</th><th style="padding: 8px; border: 1px solid #ddd;">URL</th><th style="padding: 8px; border: 1px solid #ddd;">Created</th></tr>';

                data.urls.forEach(url => {
                    html += '<tr>';
                    html += '<td style="padding: 8px; border: 1px solid #ddd;">' + url.id + '</td>';
                    html += '<td style="padding: 8px; border: 1px solid #ddd;">' + url.title + '</td>';
                    html += '<td style="padding: 8px; border: 1px solid #ddd;"><a href="' + url.url + '" target="_blank">' + url.url + '</a></td>';
                    html += '<td style="padding: 8px; border: 1px solid #ddd;">' + url.created_at + '</td>';
                    html += '</tr>';
                });

                html += '</table>';
            }

            showResult('listResult', html, true);
        } else {
            showResult('listResult', '❌ ERROR: ' + data.message, false);
        }
    } catch (error) {
        console.error('List URLs API Error:', error);
        showResult('listResult', '❌ EXCEPTION: ' + error.message, false);
    }
}

console.log('Test page loaded. PROJECT_ID =', PROJECT_ID);
console.log('Open your browser console (F12) to see detailed API responses');
</script>

<style>
body {
    font-family: Arial, sans-serif;
    max-width: 1000px;
    margin: 20px auto;
    padding: 20px;
}
h2, h3, h4 {
    color: #333;
}
input, textarea {
    font-family: Arial, sans-serif;
}
</style>
