<?php
/**
 * Configuration Test Page
 * Check if all components are working correctly
 */

echo "<style>
body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
.success { background: #d4edda; color: #155724; padding: 15px; margin: 10px 0; border-radius: 5px; border: 1px solid #c3e6cb; }
.error { background: #f8d7da; color: #721c24; padding: 15px; margin: 10px 0; border-radius: 5px; border: 1px solid #f5c6cb; }
.info { background: #d1ecf1; color: #0c5460; padding: 15px; margin: 10px 0; border-radius: 5px; border: 1px solid #bee5eb; }
h2 { color: #333; margin-top: 20px; }
pre { background: #fff; padding: 10px; border-radius: 5px; overflow-x: auto; }
</style>";

echo "<h1>🔧 Project Manager - Configuration Test</h1>";

// Test 1: PHP Version
echo "<h2>1. PHP Version</h2>";
$phpVersion = phpversion();
if (version_compare($phpVersion, '7.4.0', '>=')) {
    echo "<div class='success'>✅ PHP Version: $phpVersion (OK)</div>";
} else {
    echo "<div class='error'>❌ PHP Version: $phpVersion (Requires 7.4+)</div>";
}

// Test 2: Required Extensions
echo "<h2>2. Required PHP Extensions</h2>";
$extensions = ['pdo', 'pdo_mysql', 'session', 'json', 'mbstring'];
foreach ($extensions as $ext) {
    if (extension_loaded($ext)) {
        echo "<div class='success'>✅ Extension '$ext' is loaded</div>";
    } else {
        echo "<div class='error'>❌ Extension '$ext' is NOT loaded</div>";
    }
}

// Test 3: Config File
echo "<h2>3. Configuration File</h2>";
try {
    require_once 'includes/config.php';
    echo "<div class='success'>✅ Config file loaded successfully</div>";
    echo "<div class='info'>";
    echo "APP_NAME: " . APP_NAME . "<br>";
    echo "DB_HOST: " . DB_HOST . "<br>";
    echo "DB_NAME: " . DB_NAME . "<br>";
    echo "DB_USER: " . DB_USER . "<br>";
    echo "</div>";
} catch (Exception $e) {
    echo "<div class='error'>❌ Config file error: " . $e->getMessage() . "</div>";
}

// Test 4: Database Connection
echo "<h2>4. Database Connection</h2>";
try {
    require_once 'includes/db.php';
    $db = getDB();
    echo "<div class='success'>✅ Database connected successfully</div>";

    // Check if tables exist
    $tables = ['users', 'projects', 'tasks', 'documents', 'activity_logs'];
    echo "<div class='info'><strong>Checking tables:</strong><br>";
    foreach ($tables as $table) {
        $stmt = $db->query("SHOW TABLES LIKE '$table'");
        if ($stmt->rowCount() > 0) {
            echo "✅ Table '$table' exists<br>";
        } else {
            echo "❌ Table '$table' NOT found<br>";
        }
    }
    echo "</div>";

} catch (Exception $e) {
    echo "<div class='error'>❌ Database error: " . $e->getMessage() . "</div>";
    echo "<div class='info'>Make sure you've imported database.sql into phpMyAdmin</div>";
}

// Test 5: Auth Functions
echo "<h2>5. Authentication Functions</h2>";
try {
    require_once 'includes/auth.php';
    echo "<div class='success'>✅ Auth functions loaded</div>";
    if (function_exists('loginUser')) echo "<div class='success'>✅ loginUser() function exists</div>";
    if (function_exists('registerUser')) echo "<div class='success'>✅ registerUser() function exists</div>";
    if (function_exists('isLoggedIn')) echo "<div class='success'>✅ isLoggedIn() function exists</div>";
} catch (Exception $e) {
    echo "<div class='error'>❌ Auth functions error: " . $e->getMessage() . "</div>";
}

// Test 6: Utility Functions
echo "<h2>6. Utility Functions</h2>";
try {
    require_once 'includes/functions.php';
    echo "<div class='success'>✅ Functions loaded</div>";
    if (function_exists('e')) {
        echo "<div class='success'>✅ e() function exists</div>";
        $test = e('<script>alert("test")</script>');
        echo "<div class='info'>XSS Test: " . $test . "</div>";
    }
    if (function_exists('formatFileSize')) echo "<div class='success'>✅ formatFileSize() function exists</div>";
    if (function_exists('logActivity')) echo "<div class='success'>✅ logActivity() function exists</div>";
} catch (Exception $e) {
    echo "<div class='error'>❌ Functions error: " . $e->getMessage() . "</div>";
}

// Test 7: File Permissions
echo "<h2>7. File Permissions</h2>";
$dirs = [
    'assets/uploads' => 'Document upload directory',
    'logs' => 'Error log directory'
];

foreach ($dirs as $dir => $desc) {
    $fullPath = __DIR__ . '/' . $dir;
    if (is_dir($fullPath)) {
        if (is_writable($fullPath)) {
            echo "<div class='success'>✅ $desc ($dir) is writable</div>";
        } else {
            echo "<div class='error'>❌ $desc ($dir) is NOT writable</div>";
        }
    } else {
        echo "<div class='error'>❌ $desc ($dir) does NOT exist</div>";
        if (!is_dir($fullPath)) {
            mkdir($fullPath, 0755, true);
            echo "<div class='info'>📁 Created directory: $dir</div>";
        }
    }
}

// Test 8: Error Log
echo "<h2>8. Error Logging</h2>";
if (file_exists('logs/error.log')) {
    echo "<div class='success'>✅ Error log file exists</div>";
    if (is_writable('logs/error.log')) {
        echo "<div class='success'>✅ Error log is writable</div>";

        // Show last 20 lines of error log
        $lines = file('logs/error.log');
        if (count($lines) > 0) {
            echo "<div class='info'><strong>Last 20 error log entries:</strong><br><pre>";
            echo htmlspecialchars(implode('', array_slice($lines, -20)));
            echo "</pre></div>";
        } else {
            echo "<div class='info'>Error log is empty (no errors yet)</div>";
        }
    } else {
        echo "<div class='error'>❌ Error log is NOT writable</div>";
    }
} else {
    echo "<div class='error'>❌ Error log file does NOT exist</div>";
}

// Test 9: Demo User
echo "<h2>9. Demo User Account</h2>";
try {
    $stmt = $db->prepare("SELECT id, username, email FROM users WHERE username = 'demo'");
    $stmt->execute();
    $demoUser = $stmt->fetch();

    if ($demoUser) {
        echo "<div class='success'>✅ Demo user exists</div>";
        echo "<div class='info'>";
        echo "Username: <strong>demo</strong><br>";
        echo "Password: <strong>demo123</strong><br>";
        echo "Email: " . htmlspecialchars($demoUser['email']);
        echo "</div>";
    } else {
        echo "<div class='error'>❌ Demo user NOT found</div>";
        echo "<div class='info'>Run the database.sql script to create the demo user</div>";
    }
} catch (Exception $e) {
    echo "<div class='error'>❌ Could not check demo user: " . $e->getMessage() . "</div>";
}

// Test 10: Session
echo "<h2>10. Session Test</h2>";
if (session_status() === PHP_SESSION_ACTIVE) {
    echo "<div class='success'>✅ Session is active</div>";
    echo "<div class='info'>Session ID: " . session_id() . "</div>";
} else {
    echo "<div class='error'>❌ Session is NOT active</div>";
}

// Summary
echo "<h2>📋 Summary</h2>";
echo "<div class='info'>";
echo "<strong>If all tests pass:</strong><br>";
echo "1. Go to <a href='login.php'>login.php</a> to login<br>";
echo "2. Use demo/demo123 or register a new account<br>";
echo "3. Check <a href='logs/error.log'>error log</a> if you encounter issues<br><br>";
echo "<strong>If tests fail:</strong><br>";
echo "1. Check the error messages above<br>";
echo "2. Import database.sql in phpMyAdmin<br>";
echo "3. Ensure MySQL is running in XAMPP<br>";
echo "4. Check file permissions for uploads and logs folders<br>";
echo "</div>";
?>
