<?php
/**
 * Error Log Viewer
 * View and clear error logs
 */

$logFile = 'logs/error.log';
$action = $_GET['action'] ?? '';

// Clear log if requested
if ($action === 'clear' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    file_put_contents($logFile, "Error Log Cleared at " . date('Y-m-d H:i:s') . "\n" . str_repeat('=', 50) . "\n");
    header('Location: view-errors.php');
    exit;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Error Log Viewer - Project Manager</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.2/font/bootstrap-icons.css">
    <style>
        body { background: #f8f9fa; }
        .log-content {
            background: #1e1e1e;
            color: #d4d4d4;
            padding: 20px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            max-height: 600px;
            overflow-y: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        .error-line { color: #f48771; }
        .warning-line { color: #dcdcaa; }
        .info-line { color: #4ec9b0; }
        .timestamp { color: #9cdcfe; }
        .nav-buttons { position: sticky; top: 0; z-index: 100; background: #f8f9fa; padding: 10px 0; }
    </style>
</head>
<body>
    <div class="container py-4">
        <div class="nav-buttons">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h1><i class="bi bi-file-text"></i> Error Log Viewer</h1>
                <div>
                    <a href="test-config.php" class="btn btn-info me-2">
                        <i class="bi bi-gear"></i> Run Diagnostics
                    </a>
                    <a href="login.php" class="btn btn-primary me-2">
                        <i class="bi bi-box-arrow-in-right"></i> Go to Login
                    </a>
                    <button onclick="refreshLog()" class="btn btn-secondary me-2">
                        <i class="bi bi-arrow-clockwise"></i> Refresh
                    </button>
                    <button onclick="if(confirm('Clear all error logs?')) document.getElementById('clearForm').submit();"
                            class="btn btn-danger">
                        <i class="bi bi-trash"></i> Clear Log
                    </button>
                </div>
            </div>
        </div>

        <form id="clearForm" method="POST" action="view-errors.php?action=clear" style="display:none;"></form>

        <div class="card shadow-sm">
            <div class="card-header bg-dark text-white">
                <h5 class="mb-0">
                    <i class="bi bi-terminal"></i> logs/error.log
                    <span class="badge bg-secondary float-end" id="lineCount">0 lines</span>
                </h5>
            </div>
            <div class="card-body p-0">
                <div class="log-content" id="logContent">
                    <?php
                    if (file_exists($logFile)) {
                        $content = file_get_contents($logFile);
                        if (empty(trim($content))) {
                            echo "<div class='text-success'>✅ No errors logged yet. System is working correctly!</div>";
                        } else {
                            // Highlight different types of log entries
                            $lines = explode("\n", $content);
                            echo "<div class='info-line'>Total lines: " . count($lines) . "</div>\n";
                            echo str_repeat('-', 80) . "\n";

                            foreach ($lines as $line) {
                                if (empty(trim($line))) continue;

                                // Color code different types of messages
                                if (stripos($line, 'error') !== false || stripos($line, 'fatal') !== false) {
                                    echo "<span class='error-line'>" . htmlspecialchars($line) . "</span>\n";
                                } elseif (stripos($line, 'warning') !== false) {
                                    echo "<span class='warning-line'>" . htmlspecialchars($line) . "</span>\n";
                                } elseif (preg_match('/^\[[\d\-: ]+\]/', $line)) {
                                    echo "<span class='timestamp'>" . htmlspecialchars($line) . "</span>\n";
                                } else {
                                    echo htmlspecialchars($line) . "\n";
                                }
                            }
                        }
                    } else {
                        echo "<div class='text-danger'>❌ Error log file does not exist. It will be created automatically when an error occurs.</div>";
                    }
                    ?>
                </div>
            </div>
            <div class="card-footer text-muted">
                <small>
                    <i class="bi bi-info-circle"></i>
                    Log file location: <code><?php echo realpath($logFile) ?: $logFile; ?></code>
                    <?php if (file_exists($logFile)): ?>
                        | Size: <?php echo round(filesize($logFile) / 1024, 2); ?> KB
                        | Last modified: <?php echo date('Y-m-d H:i:s', filemtime($logFile)); ?>
                    <?php endif; ?>
                </small>
            </div>
        </div>

        <div class="alert alert-info mt-4">
            <h5><i class="bi bi-lightbulb"></i> Tips:</h5>
            <ul class="mb-0">
                <li><strong>Red lines</strong> indicate errors that need attention</li>
                <li><strong>Yellow lines</strong> indicate warnings</li>
                <li><strong>Blue lines</strong> show timestamps and general info</li>
                <li>Error logs help identify issues with database connections, file permissions, or PHP errors</li>
                <li>Run <a href="test-config.php">Diagnostics</a> to check system configuration</li>
            </ul>
        </div>

        <div class="card shadow-sm mt-4">
            <div class="card-header bg-success text-white">
                <h5 class="mb-0"><i class="bi bi-check-circle"></i> Quick Fixes</h5>
            </div>
            <div class="card-body">
                <h6>Common Issues:</h6>
                <ol>
                    <li><strong>Database Connection Failed:</strong>
                        <ul>
                            <li>Make sure MySQL is running in XAMPP</li>
                            <li>Import database.sql in phpMyAdmin</li>
                            <li>Check database credentials in includes/config.php</li>
                        </ul>
                    </li>
                    <li><strong>Blank Page / Input Not Showing:</strong>
                        <ul>
                            <li>Check error log above for PHP errors</li>
                            <li>Run <a href="test-config.php">Diagnostics</a> to identify issues</li>
                            <li>Clear browser cache and refresh</li>
                        </ul>
                    </li>
                    <li><strong>File Upload Failed:</strong>
                        <ul>
                            <li>Ensure assets/uploads/ directory exists and is writable</li>
                            <li>Check MAX_FILE_SIZE in includes/config.php</li>
                        </ul>
                    </li>
                </ol>
            </div>
        </div>
    </div>

    <script>
        function refreshLog() {
            location.reload();
        }

        // Auto-refresh every 10 seconds
        setTimeout(function() {
            const autoRefresh = confirm('Auto-refresh error log?');
            if (autoRefresh) {
                setInterval(refreshLog, 10000);
            }
        }, 1000);

        // Scroll to bottom of log
        const logContent = document.getElementById('logContent');
        logContent.scrollTop = logContent.scrollHeight;

        // Count lines
        const lines = logContent.innerText.split('\n').filter(line => line.trim()).length;
        document.getElementById('lineCount').textContent = lines + ' lines';
    </script>
</body>
</html>
